<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Ranking;
use App\Models\Transaction;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AutoAchievementDistribution extends Command
{
    protected $signature = 'achievement:auto-distribute';
    protected $description = 'Automatically distribute achievement rewards (bonus and salary) to eligible users';

    public function handle()
    {
        $timezone = setting('site_timezone', 'global') ?? 'Asia/Karachi';
        $now = Carbon::now($timezone);
        
        $this->info("Starting automatic achievement distribution at: " . $now->toDateTimeString());
        Log::info("Auto Achievement Distribution started at: " . $now->toDateTimeString());

        $distributedCount = 0;
        $totalAmount = 0;
        $errorCount = 0;

        // Process bonus distribution
        $this->info("Processing bonus distribution...");
        $bonusResult = $this->processBonusDistribution($now);
        $distributedCount += $bonusResult['count'];
        $totalAmount += $bonusResult['amount'];
        $errorCount += $bonusResult['errors'];

        // Process automatic rank promotion
        $this->info("Processing automatic rank promotion...");
        $promotionResult = $this->processAutomaticPromotion($now);
        $distributedCount += $promotionResult['count'];
        $totalAmount += $promotionResult['amount'];
        $errorCount += $promotionResult['errors'];

        // Process salary distribution
        $this->info("Processing salary distribution...");
        $salaryResult = $this->processSalaryDistribution($now);
        $distributedCount += $salaryResult['count'];
        $totalAmount += $salaryResult['amount'];
        $errorCount += $salaryResult['errors'];

        $this->info("Achievement distribution completed!");
        $this->info("Distributed: {$distributedCount} rewards");
        $this->info("Total Amount: $" . number_format($totalAmount, 2));
        $this->info("Errors: {$errorCount}");

        Log::info("Auto Achievement Distribution completed", [
            'distributed_count' => $distributedCount,
            'total_amount' => $totalAmount,
            'error_count' => $errorCount
        ]);

        return 0;
    }

    /**
     * Process bonus distribution for newly achieved ranks
     */
    private function processBonusDistribution($now)
    {
        $distributedCount = 0;
        $totalAmount = 0;
        $errorCount = 0;

        // Get users who have achieved new ranks but haven't received bonuses
        $users = User::whereHas('rankings', function($query) {
            $query->where('completed', 1)
                  ->where('bonus_collected', 0);
        })->get();

        foreach ($users as $user) {
            try {
                DB::beginTransaction();

                $userRankings = $user->rankings()->where('completed', 1)->where('bonus_collected', 0)->get();
                
                foreach ($userRankings as $userRanking) {
                    $ranking = Ranking::find($userRanking->ranking_id);
                    
                    if ($ranking && $ranking->bonus > 0) {
                        // Add bonus to user's balance
                        $user->increment('balance', $ranking->bonus);
                        
                        // Create bonus transaction
                        Transaction::create([
                            'user_id' => $user->id,
                            'type' => TxnType::Bonus,
                            'amount' => $ranking->bonus,
                            'status' => TxnStatus::Success,
                            'description' => 'Achievement bonus for rank: ' . $ranking->name,
                            'tnx' => 'BON' . strtoupper(uniqid()),
                            'created_at' => $now,
                            'updated_at' => $now
                        ]);
                        
                        // Mark bonus as collected
                        $userRanking->update(['bonus_collected' => 1]);
                        
                        $this->info("Bonus distributed for User ID {$user->id}: $" . $ranking->bonus . " for rank: " . $ranking->name);
                        $distributedCount++;
                        $totalAmount += $ranking->bonus;
                    }
                }

                DB::commit();

            } catch (\Exception $e) {
                DB::rollBack();
                $this->error("Error processing bonus for User ID {$user->id}: " . $e->getMessage());
                Log::error("Bonus distribution error", [
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
                $errorCount++;
            }
        }

        return ['count' => $distributedCount, 'amount' => $totalAmount, 'errors' => $errorCount];
    }

    /**
     * Process automatic rank promotion
     */
    private function processAutomaticPromotion($now)
    {
        $distributedCount = 0;
        $totalAmount = 0;
        $errorCount = 0;

        // Get all users who might be eligible for promotion
        $users = User::where('status', 1)->get();

        foreach ($users as $user) {
            try {
                DB::beginTransaction();

                // Get user's current rank
                $currentRank = $user->rankings()->where('completed', 1)->orderBy('id', 'desc')->first();
                $currentRankId = $currentRank ? $currentRank->ranking_id : 0;

                // Get next rank
                $nextRank = Ranking::where('id', '>', $currentRankId)->orderBy('id')->first();
                
                if ($nextRank && $this->meetsRankRequirements($user, $nextRank)) {
                    // Check if user already has this rank marked as completed
                    $existingRanking = $user->rankings()->where('ranking_id', $nextRank->id)->first();
                    
                    if (!$existingRanking || !$existingRanking->completed) {
                        // Promote user to next rank
                        $user->rankings()->updateOrCreate(
                            ['ranking_id' => $nextRank->id],
                            [
                                'completed' => 1,
                                'bonus_collected' => 0,
                                'completed_at' => $now
                            ]
                        );
                        
                        $this->info("User ID {$user->id} promoted to rank: " . $nextRank->name);
                        $distributedCount++;
                    }
                }

                DB::commit();

            } catch (\Exception $e) {
                DB::rollBack();
                $this->error("Error processing promotion for User ID {$user->id}: " . $e->getMessage());
                Log::error("Promotion error", [
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
                $errorCount++;
            }
        }

        return ['count' => $distributedCount, 'amount' => $totalAmount, 'errors' => $errorCount];
    }

    /**
     * Process salary distribution for completed ranks
     */
    private function processSalaryDistribution($now)
    {
        $distributedCount = 0;
        $totalAmount = 0;
        $errorCount = 0;

        // Get users with completed ranks who are eligible for salary
        $users = User::whereHas('rankings', function($query) {
            $query->where('completed', 1);
        })->get();

        foreach ($users as $user) {
            try {
                DB::beginTransaction();

                $userRankings = $user->rankings()->where('completed', 1)->get();
                
                foreach ($userRankings as $userRanking) {
                    $ranking = Ranking::find($userRanking->ranking_id);
                    
                    if ($ranking && $ranking->salary > 0) {
                        // Check if salary was already distributed this month
                        $lastSalary = Transaction::where('user_id', $user->id)
                            ->where('type', TxnType::Salary)
                            ->where('description', 'like', '%' . $ranking->name . '%')
                            ->whereMonth('created_at', $now->month)
                            ->whereYear('created_at', $now->year)
                            ->first();
                        
                        if (!$lastSalary) {
                            // Add salary to user's balance
                            $user->increment('balance', $ranking->salary);
                            
                            // Create salary transaction
                            Transaction::create([
                                'user_id' => $user->id,
                                'type' => TxnType::Salary,
                                'amount' => $ranking->salary,
                                'status' => TxnStatus::Success,
                                'description' => 'Monthly salary for rank: ' . $ranking->name,
                                'tnx' => 'SAL' . strtoupper(uniqid()),
                                'created_at' => $now,
                                'updated_at' => $now
                            ]);
                            
                            $this->info("Salary distributed for User ID {$user->id}: $" . $ranking->salary . " for rank: " . $ranking->name);
                            $distributedCount++;
                            $totalAmount += $ranking->salary;
                        }
                    }
                }

                DB::commit();

            } catch (\Exception $e) {
                DB::rollBack();
                $this->error("Error processing salary for User ID {$user->id}: " . $e->getMessage());
                Log::error("Salary distribution error", [
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
                $errorCount++;
            }
        }

        return ['count' => $distributedCount, 'amount' => $totalAmount, 'errors' => $errorCount];
    }

    /**
     * Check if user meets rank requirements
     */
    private function meetsRankRequirements($user, $ranking)
    {
        // Check minimum referrals
        if ($ranking->min_referral > 0) {
            $referralCount = $user->referrals()->count();
            if ($referralCount < $ranking->min_referral) {
                return false;
            }
        }

        // Check team investment
        if ($ranking->min_team_investment > 0) {
            $teamInvestment = $this->calculateTeamInvestment($user);
            if ($teamInvestment < $ranking->min_team_investment) {
                return false;
            }
        }

        // Check direct referral deposits
        if ($ranking->min_direct_deposit > 0) {
            $directDeposits = $this->calculateReferralDeposits($user);
            if ($directDeposits < $ranking->min_direct_deposit) {
                return false;
            }
        }

        // Check direct referral investments
        if ($ranking->min_direct_investment > 0) {
            $directInvestments = $this->calculateReferralInvestments($user);
            if ($directInvestments < $ranking->min_direct_investment) {
                return false;
            }
        }

        return true;
    }

    /**
     * Calculate team investment (all levels)
     */
    private function calculateTeamInvestment($user)
    {
        $totalInvestment = 0;
        
        // Get all referrals recursively
        $referrals = $user->referrals;
        foreach ($referrals as $referral) {
            $totalInvestment += $referral->invests()->where('status', 'ongoing')->sum('invest_amount');
            $totalInvestment += $this->calculateTeamInvestment($referral);
        }
        
        return $totalInvestment;
    }

    /**
     * Calculate direct referral deposits
     */
    private function calculateReferralDeposits($user)
    {
        return $user->referrals()->whereHas('deposits', function($query) {
            $query->where('status', 'success');
        })->sum('deposits.amount');
    }

    /**
     * Calculate direct referral investments
     */
    private function calculateReferralInvestments($user)
    {
        return $user->referrals()->whereHas('invests', function($query) {
            $query->where('status', 'ongoing');
        })->sum('invests.invest_amount');
    }
}













