<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RoiSchedule;
use App\Models\Invest;
use Carbon\Carbon;

class FixRoiSchedulesLogic extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'roi:fix-logic';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix ROI schedules logic - ensure proper status and timing';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Fixing ROI schedules logic...');
        
        // 1. Fix expired pending schedules to ready
        $expiredSchedules = RoiSchedule::where('status', 'pending')
            ->where('roi_end_time', '<=', now())
            ->get();
            
        $this->info("Found {$expiredSchedules->count()} expired pending schedules");
        
        foreach ($expiredSchedules as $schedule) {
            $schedule->update([
                'status' => 'ready',
                'roi_ready_time' => now()
            ]);
            $this->line("Marked Schedule ID: {$schedule->id} as ready");
        }
        
        // 2. Fix multiple pending schedules for same investment (keep only first one)
        $investments = Invest::where('status', 'ongoing')->get();
        
        foreach ($investments as $investment) {
            $pendingSchedules = RoiSchedule::where('invest_id', $investment->id)
                ->where('status', 'pending')
                ->orderBy('roi_cycle')
                ->get();
                
            if ($pendingSchedules->count() > 1) {
                $this->line("Investment ID: {$investment->id} has {$pendingSchedules->count()} pending schedules");
                
                // Keep only the first one, delete others
                $firstSchedule = $pendingSchedules->first();
                $othersToDelete = $pendingSchedules->skip(1);
                
                foreach ($othersToDelete as $schedule) {
                    $schedule->delete();
                    $this->line("Deleted duplicate Schedule ID: {$schedule->id}");
                }
            }
        }
        
        // 3. Fix ready schedules that should be sent (if admin already sent ROI)
        $readySchedules = RoiSchedule::where('status', 'ready')->get();
        
        foreach ($readySchedules as $schedule) {
            // Check if user has received ROI for this cycle
            $investment = $schedule->invest;
            if ($investment && $investment->user) {
                // Check if user's profit balance increased recently (indicating ROI was sent)
                $recentTransactions = \App\Models\Transaction::where('user_id', $investment->user_id)
                    ->where('type', 'interest')
                    ->where('created_at', '>=', $schedule->roi_ready_time)
                    ->where('amount', $schedule->roi_amount)
                    ->first();
                    
                if ($recentTransactions) {
                    $schedule->update([
                        'status' => 'sent',
                        'roi_sent_time' => $recentTransactions->created_at
                    ]);
                    $this->line("Marked Schedule ID: {$schedule->id} as sent (found matching transaction)");
                }
            }
        }
        
        // 4. Create missing next cycles for sent schedules
        $sentSchedules = RoiSchedule::where('status', 'sent')
            ->where('roi_cycle', '<', \DB::raw('total_cycles'))
            ->get();
            
        foreach ($sentSchedules as $schedule) {
            // Check if next cycle already exists
            $nextCycle = RoiSchedule::where('invest_id', $schedule->invest_id)
                ->where('roi_cycle', $schedule->roi_cycle + 1)
                ->first();
                
            if (!$nextCycle) {
                $newSchedule = RoiSchedule::create([
                    'invest_id' => $schedule->invest_id,
                    'user_id' => $schedule->user_id,
                    'schema_id' => $schedule->schema_id,
                    'invest_amount' => $schedule->invest_amount,
                    'roi_percentage' => $schedule->roi_percentage,
                    'roi_amount' => $schedule->roi_amount,
                    'roi_start_time' => $schedule->roi_sent_time ?? now(),
                    'roi_end_time' => ($schedule->roi_sent_time ?? now())->addHours(24),
                    'status' => 'pending',
                    'roi_cycle' => $schedule->roi_cycle + 1,
                    'total_cycles' => $schedule->total_cycles
                ]);
                $this->line("Created next cycle Schedule ID: {$newSchedule->id} for Investment ID: {$schedule->invest_id}");
            }
        }
        
        // 5. Fix completed schedules
        $completedSchedules = RoiSchedule::where('roi_cycle', '>=', \DB::raw('total_cycles'))
            ->where('status', '!=', 'completed')
            ->get();
            
        foreach ($completedSchedules as $schedule) {
            $schedule->update(['status' => 'completed']);
            $this->line("Marked Schedule ID: {$schedule->id} as completed");
        }
        
        $this->info('ROI schedules logic fixed successfully!');
        
        return Command::SUCCESS;
    }
}
