<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Illuminate\Support\Facades\Storage;

class GenerateUserAvatars extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'users:generate-avatars {--user-id= : Generate avatar for specific user ID} {--all : Generate avatars for all users}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate avatars for users';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🎨 Starting avatar generation...');

        if ($userId = $this->option('user-id')) {
            $this->generateAvatarForUser($userId);
        } elseif ($this->option('all')) {
            $this->generateAvatarsForAllUsers();
        } else {
            $this->error('Please specify --user-id or --all option');
            return 1;
        }

        return 0;
    }

    private function generateAvatarForUser($userId)
    {
        $user = User::find($userId);
        if (!$user) {
            $this->error("User with ID {$userId} not found!");
            return;
        }

        $this->info("Generating avatar for user: {$user->username}");

        try {
            $avatarPath = $this->generateAvatar($user->username, $user->id);
            
            // Update user's avatar
            $user->avatar = $avatarPath;
            $user->save();

            $this->info("✅ Avatar generated successfully for {$user->username}");
            $this->info("Avatar path: {$avatarPath}");

        } catch (\Exception $e) {
            $this->error("❌ Failed to generate avatar for {$user->username}: " . $e->getMessage());
        }
    }

    private function generateAvatarsForAllUsers()
    {
        $users = User::all();
        $totalUsers = $users->count();
        $generatedCount = 0;
        $errors = [];

        $this->info("Found {$totalUsers} users. Starting avatar generation...");

        $progressBar = $this->output->createProgressBar($totalUsers);
        $progressBar->start();

        foreach ($users as $user) {
            try {
                $avatarPath = $this->generateAvatar($user->username, $user->id);
                $user->avatar = $avatarPath;
                $user->save();
                $generatedCount++;
            } catch (\Exception $e) {
                $errors[] = "User {$user->username}: " . $e->getMessage();
            }
            
            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine();

        $this->info("✅ Avatar generation completed!");
        $this->info("Generated: {$generatedCount}/{$totalUsers} users");

        if (!empty($errors)) {
            $this->warn("⚠️  Errors encountered:");
            foreach ($errors as $error) {
                $this->line("   - {$error}");
            }
        }
    }

    private function generateAvatar($username, $userId)
    {
        // Create avatar directory if it doesn't exist
        $avatarDir = 'public/avatars';
        if (!Storage::exists($avatarDir)) {
            Storage::makeDirectory($avatarDir);
        }

        // Generate unique filename
        $filename = 'avatar_' . $userId . '_' . time() . '.svg';
        $filepath = $avatarDir . '/' . $filename;

        // Generate SVG avatar
        $svg = $this->generateSVGAvatar($username);
        
        // Save SVG file
        Storage::put($filepath, $svg);

        // Return the public URL - fix the path to match storage link
        return '/storage/avatars/' . $filename;
    }

    private function generateSVGAvatar($username)
    {
        // Get first letter of username
        $initial = strtoupper(substr($username, 0, 1));
        
        // Generate random colors
        $colors = [
            '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
            '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9',
            '#F8C471', '#82E0AA', '#F1948A', '#85C1E9', '#D7BDE2'
        ];
        $bgColor = $colors[array_rand($colors)];
        
        // Generate complementary text color
        $textColor = $this->getContrastColor($bgColor);

        $svg = '<?xml version="1.0" encoding="UTF-8"?>
        <svg width="100" height="100" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
            <defs>
                <linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="100%">
                    <stop offset="0%" style="stop-color:' . $bgColor . ';stop-opacity:1" />
                    <stop offset="100%" style="stop-color:' . $this->darkenColor($bgColor, 20) . ';stop-opacity:1" />
                </linearGradient>
            </defs>
            <circle cx="50" cy="50" r="50" fill="url(#grad)" />
            <text x="50" y="65" font-family="Arial, sans-serif" font-size="40" font-weight="bold" text-anchor="middle" fill="' . $textColor . '">' . $initial . '</text>
        </svg>';

        return $svg;
    }

    private function getContrastColor($hexColor)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Calculate luminance
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
        
        // Return black or white based on luminance
        return $luminance > 0.5 ? '#000000' : '#FFFFFF';
    }

    private function darkenColor($hexColor, $percent)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Darken by percentage
        $r = max(0, min(255, $r - ($r * $percent / 100)));
        $g = max(0, min(255, $g - ($g * $percent / 100)));
        $b = max(0, min(255, $b - ($b * $percent / 100)));
        
        // Convert back to hex
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }
}
