<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\User;
use App\Notifications\InvestmentAlert;
use Illuminate\Support\Facades\Notification;

class MonitorInvestments extends Command
{
    protected $signature = 'invest:monitor';
    protected $description = 'Monitor stuck investments and trigger recovery';

    public function handle()
    {
        // 1. Find stuck investments (profit time overdue by >24 hours)
        $stuckInvestments = Invest::where('status', 'Ongoing')
            ->where('next_profit_time', '<', now()->subDay())
            ->with('user')
            ->get();

        if ($stuckInvestments->isEmpty()) {
            $this->info('No stuck investments found.');
            return;
        }

        // 2. Send alert to admins
        $adminUsers = User::where('is_admin', true)->get();
        
        if ($adminUsers->isNotEmpty()) {
            Notification::send($adminUsers, new InvestmentAlert(
                count: $stuckInvestments->count(),
                exampleId: $stuckInvestments->first()->id
            ));
            $this->info('Sent alerts to admin users.');
        }

        // 3. Attempt automatic recovery
        try {
            $this->call('invest:recover-missed-profits');
            
            // Log successful recovery
            \Log::channel('investments')->info(
                'Recovered '.$stuckInvestments->count().' stuck investments',
                $stuckInvestments->pluck('id')->toArray()
            );
            
            $this->info('Successfully triggered profit recovery.');
            
        } catch (\Exception $e) {
            \Log::channel('investments')->error(
                'Failed to recover stuck investments: '.$e->getMessage()
            );
            
            $this->error('Recovery failed: '.$e->getMessage());
        }
    }
}