<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Page;
use App\Traits\NotifyTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Str;

class PasswordResetLinkController extends Controller
{
    use NotifyTrait;

    /**
     * Display the password reset link request view.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $page = Page::where('code', 'forgetpassword')->first();
        $data = json_decode($page->data, true);

        return view('frontend.shahdeveloper.auth.forgot-password', compact('data'));
    }

    /**
     * Handle an incoming password reset link request.
     *
     * @return \Illuminate\Http\RedirectResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request)
    {
        // Validate email exists in users table
        $validator = Validator::make(
            $request->all(),
            [
                'email' => 'required|email|exists:users,email',
            ],
            [
                'email.required' => __('Please enter your email address.'),
                'email.email' => __('Please enter a valid email address.'),
                'email.exists' => __('We can\'t find a user with that email address.'),
            ]
        );

        if ($validator->fails()) {
            if ($request->expectsJson()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                    'errors' => $validator->errors(),
                ], 422);
            }
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Use Laravel Password Broker → handles tokens in password_reset_tokens and mailing
        $status = Password::sendResetLink($request->only('email'));

        if ($status === Password::RESET_LINK_SENT) {
            if ($request->expectsJson()) {
                return response()->json(['message' => trans($status)]);
            }
            return back()->with('status', trans($status));
        }

        if ($request->expectsJson()) {
            return response()->json(['message' => trans($status)], 500);
        }
        notify()->error(trans($status), 'Error');
        return back()->withErrors(['email' => trans($status)]);
    }
}
