<?php

namespace App\Http\Controllers\Backend;

use App\Enums\GatewayType;
use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Http\Controllers\Controller;
use App\Models\DepositMethod;
use App\Models\Gateway;
use App\Models\Invest;
use App\Models\LevelReferral;
use App\Models\Transaction;
use App\Traits\ImageUpload;
use App\Traits\NotifyTrait;
use Carbon\Carbon;
use DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Purifier;
use Txn;

class DepositController extends Controller
{
    use NotifyTrait, ImageUpload;

    /**
     * Display a listing of the resource.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:deposit-list|deposit-action', ['only' => ['pending', 'history']]);
        $this->middleware('permission:deposit-action', ['only' => ['depositAction', 'actionNow']]);
    }

    //-------------------------------------------  Deposit method start ---------------------------------------------------------------

    public function methodList($type)
    {
        $button = [
            'name' => __('ADD NEW'),
            'icon' => 'plus',
            'route' => route('admin.deposit.method.create', $type),
        ];

        $depositMethods = DepositMethod::where('type', $type)->get();

        return view('backend.deposit.method_list', compact('depositMethods', 'button', 'type'));
    }

    public function createMethod($type)
    {
        $gateways = Gateway::where('status', true)->get();

        return view('backend.deposit.create_method', compact('type', 'gateways'));
    }

    public function methodStore(Request $request)
    {
        $input = $request->all();

        // Enhanced validation with better error messages
        $validator = Validator::make($input, [
            'logo' => 'nullable',
            'name' => 'required|max:255',
            'gateway_id' => 'nullable',
            'method_code' => 'nullable|unique:deposit_methods,gateway_code',
            'currency' => 'required|max:10',
            'currency_symbol' => 'required|max:10',
            'charge' => 'required|numeric|min:0',
            'charge_type' => 'required|in:percentage,fixed',
            'rate' => 'required|numeric|min:0',
            'minimum_deposit' => 'required|numeric|min:0',
            'maximum_deposit' => 'required|numeric|min:0|gte:minimum_deposit',
            'status' => 'required|in:0,1',
            'field_options' => 'nullable|array',
        ], [
            'name.required' => 'Method name is required',
            'method_code.unique' => 'This method code is already in use',
            'currency.required' => 'Currency is required',
            'currency_symbol.required' => 'Currency symbol is required',
            'charge.required' => 'Charge amount is required',
            'charge.numeric' => 'Charge must be a valid number',
            'charge_type.required' => 'Charge type is required',
            'rate.required' => 'Conversion rate is required',
            'rate.numeric' => 'Rate must be a valid number',
            'minimum_deposit.required' => 'Minimum deposit amount is required',
            'minimum_deposit.numeric' => 'Minimum deposit must be a valid number',
            'maximum_deposit.required' => 'Maximum deposit amount is required',
            'maximum_deposit.numeric' => 'Maximum deposit must be a valid number',
            'maximum_deposit.gte' => 'Maximum deposit must be greater than or equal to minimum deposit',
            'status.required' => 'Status is required',
            'field_options.array' => 'Field options must be an array',
        ]);

        // After: enforce minimal requirements based on type without required_if
        $validator->after(function ($v) use ($input) {
            if (($input['type'] ?? null) === 'auto' && empty($input['gateway_id'])) {
                $v->errors()->add('gateway_id', 'Gateway selection is required for automatic methods');
            }
            if (($input['type'] ?? null) === 'manual') {
                // Accept either method_code or at least one field option for manual configuration
                $hasMethodCode = !empty($input['method_code']);
                $hasFieldOptions = !empty($input['field_options']) && is_array($input['field_options']) && count($input['field_options']) >= 1;
                if (!$hasMethodCode && !$hasFieldOptions) {
                    $v->errors()->add('method_code', 'Provide a method code or at least one field option for manual methods');
                }
            }
        });

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $errorMessage = 'Validation failed: ' . implode(', ', $errors);
            notify()->error($errorMessage, 'Error');

            return redirect()->back()->withInput();
        }

        if (isset($input['gateway_id'])) {
            $gateway = Gateway::find($input['gateway_id']);
            $methodCode = $gateway->gateway_code . '-' . strtolower($input['currency']);
        }

        $data = [
            'logo' => isset($input['logo']) ? self::imageUploadTrait($input['logo']) : null,
            'name' => $input['name'],
            'type' => $input['type'],
            'gateway_id' => $input['gateway_id'] ?? null,
            'gateway_code' => $input['method_code'] ?? $methodCode,
            'currency' => $input['currency'],
            'currency_symbol' => $input['currency_symbol'],
            'charge' => $input['charge'],
            'charge_type' => $input['charge_type'],
            'rate' => $input['rate'],
            'minimum_deposit' => $input['minimum_deposit'],
            'maximum_deposit' => $input['maximum_deposit'],
            'status' => $input['status'],
            'field_options' => isset($input['field_options']) ? json_encode($input['field_options']) : null,
            // Save JSON exactly as JSON (no purifier), normalized
            'payment_details' => isset($input['payment_details']) && $input['payment_details'] !== ''
                ? json_encode(json_decode($input['payment_details'], true))
                : null,
        ];

        $depositMethod = DepositMethod::create($data);
        notify()->success($depositMethod->name . ' ' . __(' Method Created'));

        return redirect()->route('admin.deposit.method.list', $depositMethod->type);
    }

    public function methodEdit($type)
    {
        $gateways = Gateway::where('status', true)->get();
        $method = DepositMethod::find(\request('id'));
        $supported_currencies = Gateway::find($method->gateway_id)->supported_currencies ?? [];

        return view('backend.deposit.edit_method', compact('method', 'type', 'gateways', 'supported_currencies'));
    }

    public function methodUpdate($id, Request $request)
    {
        $input = $request->all();
        $validator = Validator::make($input, [
            'name' => 'required',
            'gateway_id' => 'nullable',
            'currency' => 'required',
            'currency_symbol' => 'required',
            'charge' => 'required',
            'charge_type' => 'required',
            'rate' => 'required',
            'minimum_deposit' => 'required',
            'maximum_deposit' => 'required',
            'status' => 'required',
            'field_options' => 'nullable',
        ]);

        // After: enforce minimal requirements based on type without required_if
        $validator->after(function ($v) use ($input) {
            if (($input['type'] ?? null) === 'auto' && empty($input['gateway_id'])) {
                $v->errors()->add('gateway_id', 'Gateway selection is required for automatic methods');
            }
            if (($input['type'] ?? null) === 'manual') {
                $hasFieldOptions = !empty($input['field_options']);
                // Allow manual methods without field_options since we moved to direct wallet details storage elsewhere
                // Only warn if absolutely nothing is provided (kept lenient)
            }
        });

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');

            return redirect()->back();
        }

        $depositMethod = DepositMethod::find($id);

        $user = \Auth::user();
        if ($depositMethod->type == GatewayType::Automatic) {
            if (!$user->can('automatic-gateway-manage')) {
                return redirect()->route('admin.deposit.method.list', $depositMethod->type);
            }

        } else {
            if (!$user->can('manual-gateway-manage')) {
                return redirect()->route('admin.deposit.method.list', $depositMethod->type);
            }
        }

        $data = [
            'name' => $input['name'],
            'type' => $input['type'],
            'gateway_id' => $input['gateway_id'] ?? null,
            'currency' => $input['currency'],
            'currency_symbol' => $input['currency_symbol'],
            'charge' => $input['charge'],
            'charge_type' => $input['charge_type'],
            'rate' => $input['rate'],
            'minimum_deposit' => $input['minimum_deposit'],
            'maximum_deposit' => $input['maximum_deposit'],
            'status' => $input['status'],
            'field_options' => isset($input['field_options']) ? json_encode($input['field_options']) : null,
            // Save JSON exactly as JSON (no purifier), normalized
            'payment_details' => isset($input['payment_details']) && $input['payment_details'] !== ''
                ? json_encode(json_decode($input['payment_details'], true))
                : null,
        ];

        if ($request->hasFile('logo')) {
            $logo = self::imageUploadTrait($input['logo'], $depositMethod->logo);
            $data = array_merge($data, ['logo' => $logo]);
        }

        $depositMethod->update($data);
        notify()->success($depositMethod->name . ' ' . __(' Method Updated'));

        return redirect()->route('admin.deposit.method.list', $depositMethod->type);
    }

    //-------------------------------------------  Deposit method end ---------------------------------------------------------------

    public function pending(Request $request)
    {

        if ($request->ajax()) {
            $data = Transaction::where('status', 'pending')->where(function ($query) {
                return $query->where('type', TxnType::ManualDeposit)
                    ->orWhere('type', TxnType::Investment);
            })->latest();

            return Datatables::of($data)
                ->addIndexColumn()
                ->editColumn('status', 'backend.transaction.include.__txn_status')
                ->editColumn('type', 'backend.transaction.include.__txn_type')
                ->editColumn('amount', 'backend.transaction.include.__txn_amount')
                ->editColumn('charge', function ($request) {
                    return $request->charge . ' ' . setting('site_currency', 'global');
                })
                ->addColumn('username', 'backend.transaction.include.__user')
                ->addColumn('action', 'backend.deposit.include.__action')
                ->rawColumns(['action', 'status', 'type', 'amount', 'username'])
                ->make(true);
        }

        return view('backend.deposit.manual');
    }

    public function history(Request $request)
    {

        if ($request->ajax()) {
            $data = Transaction::where(function ($query) {
                $query->where('type', TxnType::ManualDeposit)
                    ->orWhere('type', TxnType::Deposit);
            })->latest();

            return Datatables::of($data)
                ->addIndexColumn()
                ->editColumn('status', 'backend.transaction.include.__txn_status')
                ->editColumn('type', 'backend.transaction.include.__txn_type')
                ->editColumn('final_amount', 'backend.transaction.include.__txn_amount')
                ->editColumn('charge', function ($request) {
                    return $request->charge . ' ' . setting('site_currency', 'global');
                })
                ->addColumn('username', 'backend.transaction.include.__user')
                ->rawColumns(['status', 'type', 'final_amount', 'username'])
                ->make(true);
        }

        return view('backend.deposit.history');
    }

    public function depositAction($id)
    {

        $data = Transaction::find($id);
        return view('backend.deposit.include.__deposit_action', compact('data', 'id'))->render();
    }

    public function actionNow(Request $request)
    {

        $input = $request->all();
        $id = $input['id'];
        $approvalCause = $input['message'];
        $transaction = Transaction::find($id);

        if (isset($input['approve'])) {

            if ($transaction->type == TxnType::Investment) {
                $invest = Invest::where('transaction_id', $id)->first();
                $periodHours = $invest->period_hours;
                $nextProfitTime = Carbon::now()->addHour($periodHours);
                $invest->update([
                    'next_profit_time' => $nextProfitTime,
                    'status' => InvestStatus::Ongoing,
                ]);

                //level referral - REMOVED: Investment referral commission disabled
                // Only deposit referral commission is allowed (one-time)

            } else {
                $transaction->user->increment('balance', $transaction->amount);

                //level referral
                if (setting('site_referral', 'global') == 'level' && setting('deposit_level')) {
                    $level = LevelReferral::where('type', 'deposit')->max('the_order') + 1;
                    creditReferralBonus($transaction->user, 'deposit', $transaction->amount, $level);
                }
            }

            Txn::update($transaction->tnx, TxnStatus::Success, $transaction->user_id, $approvalCause);

            notify()->success('Approve successfully');

        } elseif (isset($input['reject'])) {
            $invest = Invest::where('transaction_id', $id)->first();

            if ($invest) {
                $invest->delete();
            }
            Txn::update($transaction->tnx, TxnStatus::Failed, $transaction->user_id, $approvalCause);
            notify()->success('Reject successfully');
        }

        $shortcodes = [
            '[[full_name]]' => $transaction->user->full_name,
            '[[txn]]' => $transaction->tnx,
            '[[gateway_name]]' => $transaction->method,
            '[[deposit_amount]]' => $transaction->amount,
            '[[site_title]]' => setting('site_title', 'global'),
            '[[site_url]]' => route('home'),
            '[[message]]' => $transaction->approval_cause,
            '[[status]]' => isset($input['approve']) ? 'approved' : 'Rejected',
        ];

        $this->mailNotify($transaction->user->email, 'user_manual_deposit_request', $shortcodes);
        $this->pushNotify('user_manual_deposit_request', $shortcodes, route('user.deposit.log'), $transaction->user->id);
        $this->smsNotify('user_manual_deposit_request', $shortcodes, $transaction->user->phone);

        return redirect()->back();
    }

    public function depositDetails($id)
    {
        $deposit = Transaction::with('user')->findOrFail($id);
        $manualData = json_decode($deposit->manual_data, true);
        
        return view('admin.deposit.details', compact('deposit', 'manualData'));
    }

    public function depositActionNow(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'deposit_id' => 'required|exists:transactions,id',
            'action_type' => 'required|in:approve,reject',
            'admin_notes' => 'nullable|string|max:1000',
            'payment_type' => 'nullable|in:bank_transfer,crypto',
            'account_number' => 'nullable|string|max:255',
            'transaction_id' => 'nullable|string|max:255',
            'account_name' => 'nullable|string|max:255',
            'wallet_address' => 'nullable|string|max:255',
            'transaction_hash' => 'nullable|string|max:255',
            'crypto_type' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        }

        try {
            $transaction = Transaction::findOrFail($request->deposit_id);
            
            if ($request->action_type === 'approve') {
                // Prepare payment type data
                $paymentData = [];
                if ($request->payment_type) {
                    $paymentData['payment_type'] = $request->payment_type;
                    
                    if ($request->payment_type === 'bank_transfer') {
                        $paymentData['bank_details'] = [
                            'account_number' => $request->account_number,
                            'transaction_id' => $request->transaction_id,
                            'account_name' => $request->account_name,
                        ];
                    } elseif ($request->payment_type === 'crypto') {
                        $paymentData['crypto_details'] = [
                            'wallet_address' => $request->wallet_address,
                            'transaction_hash' => $request->transaction_hash,
                            'crypto_type' => $request->crypto_type,
                        ];
                    }
                }
                
                // Approve deposit
                $transaction->user->increment('balance', $transaction->amount);
                $transaction->update([
                    'status' => TxnStatus::Success,
                    'approval_cause' => $request->admin_notes,
                    'payment_type_data' => !empty($paymentData) ? json_encode($paymentData) : null
                ]);

                // ✅ FIXED: Add referral commission for deposit approval
                if (setting('site_referral', 'global') == 'level' && setting('deposit_level')) {
                    $level = LevelReferral::where('type', 'deposit')->max('the_order') + 1;
                    creditReferralBonus($transaction->user, 'deposit', $transaction->amount, $level);
                }

                // Send notification to user
                notify()->success('Deposit approved successfully', 'Success');

                $message = 'Deposit approved successfully';
            } else {
                // Reject deposit
                $transaction->update([
                    'status' => TxnStatus::Failed,
                    'approval_cause' => $request->admin_notes
                ]);

                // Send notification to user
                notify()->success('Deposit rejected successfully', 'Success');

                $message = 'Deposit rejected successfully';
            }

            return response()->json([
                'success' => true,
                'message' => $message
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error processing request: ' . $e->getMessage()
            ]);
        }
    }
}
