<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Invest;
use App\Models\User;
use App\Models\Schema;
use App\Models\Transaction;
use App\Models\ProfitDistribution;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ProfitController extends Controller
{
    /**
     * Manual Profit Distribution Page
     */
    public function manualDistribution()
    {
        // Get all active investments
        $activeInvestments = Invest::with(['user', 'schema'])
            ->where('status', 'ongoing')
            ->orderBy('next_profit_time', 'asc')
            ->get();

        // Get all schemas for reference
        $schemas = Schema::where('status', true)->get();

        // Calculate total pending profit using dynamic ROI
        $totalPendingProfit = 0;
        foreach ($activeInvestments as $investment) {
            // Use dynamic ROI calculation instead of static interest
            $profitDue = $investment->calculated_profit;
            $totalPendingProfit += $profitDue;
        }

        return view('backend.profit.manual-distribution', compact(
            'activeInvestments',
            'schemas',
            'totalPendingProfit'
        ));
    }

    /**
     * Generate unique transaction ID
     */
    private function generateUniqueTnx()
    {
        do {
            $tnx = 'INT' . strtoupper(Str::random(10));
        } while (Transaction::where('tnx', $tnx)->exists());
        
        return $tnx;
    }

    /**
     * Distribute Profit to Specific Investment
     */
    public function distributeProfit(Request $request)
    {
        $request->validate([
            'investment_id' => 'required|exists:invests,id',
            'profit_amount' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            $investment = Invest::with(['user', 'schema'])->findOrFail($request->investment_id);
            $user = $investment->user;
            $profitAmount = $request->profit_amount;

            // Update investment using hours-based schedule (default 24h)
            $periodHours = optional($investment->schema->schedule)->time
                ?? ($investment->period_hours ?: 24);
            $nextProfitTime = Carbon::now()->addHours($periodHours);
            
            $updateData = [
                'next_profit_time' => $nextProfitTime,
                'last_profit_time' => Carbon::now(),
                'number_of_period' => ($investment->number_of_period - 1),
                'already_return_profit' => ($investment->already_return_profit + 1),
                'total_profit_amount' => ($investment->total_profit_amount + $profitAmount),
            ];

            if ($investment->return_type == 'lifetime') {
                $investment->update($updateData);
                $user->increment('profit_balance', $profitAmount);
            } else {
                if ($investment->number_of_period > 0) {
                    if ($investment->number_of_period == 1) {
                        $updateData['status'] = 'completed';
                        
                        if ($investment->capital_back == 1) {
                            $user->increment('balance', $investment->invest_amount);
                        }
                    }
                    
                    $investment->update($updateData);
                    $user->increment('profit_balance', $profitAmount);
                }
            }

            // Create transaction record with unique tnx
            $transaction = new Transaction();
            $transaction->user_id = $user->id;
            $transaction->from_model = 'Admin';
            $transaction->amount = $profitAmount;
            $transaction->type = 'interest';
            $transaction->status = 'success';
            $transaction->tnx = $this->generateUniqueTnx();
            $transaction->description = $investment->schema->name . ' Plan Interest (Manual)';
            $transaction->charge = 0;
            $transaction->final_amount = $profitAmount;
            $transaction->method = 'manual';
            $transaction->pay_currency = setting('site_currency', 'global') ?? 'USD';
            $transaction->pay_amount = $profitAmount;
            $transaction->created_at = Carbon::now();
            $transaction->updated_at = Carbon::now();
            $transaction->save();

            // Record distribution history
            $distribution = new ProfitDistribution();
            $distribution->investment_id = $investment->id;
            $distribution->user_id = $user->id;
            $distribution->profit_amount = $profitAmount;
            $distribution->distributed_by = auth()->id();
            $distribution->distribution_time = Carbon::now();
            $distribution->notes = $request->notes;
            $distribution->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Profit distributed successfully!',
                'data' => [
                    'user' => $user->username,
                    'amount' => $profitAmount,
                    'next_profit_time' => $nextProfitTime->format('Y-m-d H:i:s')
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk Profit Distribution
     */
    public function bulkDistribute(Request $request)
    {
        $request->validate([
            'investment_ids' => 'required|array',
            'investment_ids.*' => 'exists:invests,id'
        ]);

        try {
            DB::beginTransaction();

            $distributedCount = 0;
            $totalProfit = 0;
            $errors = [];

            foreach ($request->investment_ids as $investmentId) {
                try {
                    $investment = Invest::with(['user', 'schema'])->findOrFail($investmentId);
                    
                    // Check if profit is due
                    if ($investment->next_profit_time > Carbon::now()) {
                        continue;
                    }

                    // Use dynamic ROI calculation instead of static interest
                    $profitDue = $investment->calculated_profit;

                    // Update investment using hours-based schedule (default 24h)
                    $periodHours = optional($investment->schema->schedule)->time
                        ?? ($investment->period_hours ?: 24);
                    $nextProfitTime = Carbon::now()->addHours($periodHours);
                    
                    $updateData = [
                        'next_profit_time' => $nextProfitTime,
                        'last_profit_time' => Carbon::now(),
                        'number_of_period' => ($investment->number_of_period - 1),
                        'already_return_profit' => ($investment->already_return_profit + 1),
                        'total_profit_amount' => ($investment->total_profit_amount + $profitDue),
                    ];

                    if ($investment->return_type == 'lifetime') {
                        $investment->update($updateData);
                        $investment->user->increment('profit_balance', $profitDue);
                    } else {
                        if ($investment->number_of_period > 0) {
                            if ($investment->number_of_period == 1) {
                                $updateData['status'] = 'completed';
                                
                                if ($investment->capital_back == 1) {
                                    $investment->user->increment('balance', $investment->invest_amount);
                                }
                            }
                            
                            $investment->update($updateData);
                            $investment->user->increment('profit_balance', $profitDue);
                        }
                    }

                    // Create transaction with unique tnx
                    $transaction = new Transaction();
                    $transaction->user_id = $investment->user->id;
                    $transaction->from_model = 'Admin';
                    $transaction->amount = $profitDue;
                    $transaction->type = 'interest';
                    $transaction->status = 'success';
                    $transaction->tnx = $this->generateUniqueTnx();
                    $transaction->description = $investment->schema->name . ' Plan Interest (Bulk)';
                    $transaction->charge = 0;
                    $transaction->final_amount = $profitDue;
                    $transaction->method = 'bulk';
                    $transaction->pay_currency = setting('site_currency', 'global') ?? 'USD';
                    $transaction->pay_amount = $profitDue;
                    $transaction->created_at = Carbon::now();
                    $transaction->updated_at = Carbon::now();
                    $transaction->save();

                    // Record distribution
                    $distribution = new ProfitDistribution();
                    $distribution->investment_id = $investment->id;
                    $distribution->user_id = $investment->user->id;
                    $distribution->profit_amount = $profitDue;
                    $distribution->distributed_by = auth()->id();
                    $distribution->distribution_time = Carbon::now();
                    $distribution->notes = 'Bulk distribution';
                    $distribution->save();

                    $distributedCount++;
                    $totalProfit += $profitDue;

                } catch (\Exception $e) {
                    $errors[] = "Investment ID {$investmentId}: " . $e->getMessage();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Bulk distribution completed! {$distributedCount} investments processed.",
                'data' => [
                    'distributed_count' => $distributedCount,
                    'total_profit' => $totalProfit,
                    'errors' => $errors
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Bulk distribution failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Distribution History
     */
    public function distributionHistory()
    {
        $distributions = ProfitDistribution::with(['user', 'investment.schema'])
            ->orderBy('distribution_time', 'desc')
            ->paginate(50);

        return view('backend.profit.distribution-history', compact('distributions'));
    }

    /**
     * All Profits (DataTable Ajax Response)
     */
    public function allProfits($id = null)
    {
        // Check if it's Ajax request (DataTable)
        if (request()->ajax()) {
            $profits = Transaction::where('user_id', $id)
                ->where('type', 'interest')
                ->select([
                    'created_at',
                    'amount as final_amount', 
                    'type',
                    'description as profit_from',
                    'description'
                ])
                ->orderBy('created_at', 'desc');

            // Return JSON for DataTable
            return datatables()->of($profits)
                ->addIndexColumn()
                ->make(true);
        }

        return view('backend.profit.all-profits');
    }
}
