<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Schedule;
use App\Models\Schema;
use App\Traits\ImageUpload;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;

class SchemaController extends Controller
{
    use ImageUpload;

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function __construct()
    {
        $this->middleware('permission:schema-list|schema-create|schema-edit', ['only' => ['index', 'store']]);
        $this->middleware('permission:schema-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:schema-edit', ['only' => ['edit', 'update']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return Application|Factory|View
     */
    public function index()
    {
        $schemas = Schema::all();

        return view('backend.schema.index', compact('schemas'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return RedirectResponse
     */
    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
    'icon' => 'required',
    'name' => 'required',
    'type' => 'required',
    'min_amount' => 'required_if:type,==,range',
    'max_amount' => 'required_if:type,==,range',
    'fixed_amount' => 'required_if:type,==,fixed',
    'capital_back' => 'required',
    'featured' => 'required',
    'badge' => 'required_if:featured,==,1',
    'status' => 'required',
    'return_interest' => 'required|numeric|min:0',
    'interest_type' => 'required_with:return_interest|in:percentage,fixed',
    'return_period' => 'required',
    'return_type' => 'required',
    'number_of_period' => 'required_if:return_type,==,period|nullable|integer|min:0',
    'expiry_minute' => 'required_if:schema_cancel,1|nullable|integer|min:1',
    'schema_duration' => 'nullable|integer|min:0',
    'early_withdrawal_deduction' => 'nullable|numeric|min:0|max:100',
    'money_back_period' => 'nullable|integer|min:0',
]);


        if ($validator->fails()) {
            // Debug: Show all validation errors
            $errors = $validator->errors();
            notify()->error('Validation Error: ' . $errors->first(), 'Error');
            
            // Log all errors for debugging
            \Log::error('Schema Store Validation Errors:', $errors->toArray());

            return redirect()->back();
        }

        $input = $request->all();

        $finalData = [
            'name' => $input['name'],

            'type' => $input['type'],
            'min_amount' => $input['type'] == 'fixed' ? 0 : $input['min_amount'],
            'max_amount' => $input['type'] == 'fixed' ? 0 : $input['max_amount'],
            'fixed_amount' => $input['type'] == 'range' ? 0 : $input['fixed_amount'],

            'capital_back' => $input['capital_back'],

            'featured' => $input['featured'],
            'badge' => $input['featured'] == 1 ? $input['badge'] : null,

            'status' => $input['status'],
            'return_interest' => $input['return_interest'],
            'interest_type' => $input['interest_type'],
            'return_period' => $input['return_period'],
            'return_type' => $input['return_type'],
            'number_of_period' => $input['return_type'] == 'period' ? $input['number_of_period'] : 0,
            'off_days' => isset($input['off_days']) ? json_encode($input['off_days']) : null,
            'schema_duration' => $input['schema_duration'] ?? null,
            'schema_cancel' => $input['schema_cancel'],
            'expiry_minute' => $input['schema_cancel'] != 0 ? $input['expiry_minute'] : 0,
            'early_withdrawal_deduction' => $input['early_withdrawal_deduction'] ?? 25,
            'money_back_period' => $input['money_back_period'] ?? 15,
            'is_trending' => $input['is_trending'],
            'icon' => self::imageUploadTrait($input['icon']),
        ];


        Schema::create($finalData);

        notify()->success('schema created successfully');

        return redirect()->route('admin.schema.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Application|Factory|View
     */
    public function create()
    {
        $schedules = Schedule::all();

        $offDaySchedule = [
            'Sunday',
            'Monday',
            'Tuesday',
            'Wednesday',
            'Thursday',
            'Friday',
            'Saturday',
        ];

        return view('backend.schema.create', compact('schedules', 'offDaySchedule'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return Application|Factory|View
     */
    public function edit($id)
    {

        $schedules = Schedule::all();
        $schema = Schema::find($id);

        $offDaySchedule = [
            'Sunday',
            'Monday',
            'Tuesday',
            'Wednesday',
            'Thursday',
            'Friday',
            'Saturday',
        ];

        return view('backend.schema.edit', compact('schema', 'schedules', 'offDaySchedule'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param int $id
     * @return RedirectResponse
     */
    public function update(Request $request, $id)
    {
        // Debug: Log all request data
        \Log::info('Schema Update Request Data:', $request->all());
        
        // Debug: Check specific fields
        \Log::info('Return Interest: ' . $request->input('return_interest'));
        \Log::info('Number of Period: ' . $request->input('number_of_period'));
        \Log::info('Schema Duration: ' . $request->input('schema_duration'));
        \Log::info('Early Withdrawal Deduction: ' . $request->input('early_withdrawal_deduction'));
        \Log::info('Money Back Period: ' . $request->input('money_back_period'));

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'type' => 'required',
            'min_amount' => 'required_if:type,==,range',
            'max_amount' => 'required_if:type,==,range',
            'fixed_amount' => 'required_if:type,==,fixed',
            'capital_back' => 'required',
            'featured' => 'required',
            'badge' => 'required_if:featured,==,1',
            'status' => 'required',
            'return_interest' => 'required|numeric',
            'interest_type' => 'required',
            'return_period' => 'required',
            'return_type' => 'required',
            'number_of_period' => 'required_if:return_type,==,period|nullable|integer',
            'expiry_minute' => 'required_if:schema_cancel,1|nullable|integer',
            'schema_duration' => 'nullable|integer',
            'early_withdrawal_deduction' => 'nullable|numeric',
            'money_back_period' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            // Debug: Show all validation errors
            $errors = $validator->errors();
            
            // Show detailed error for each field
            $errorMessages = [];
            foreach ($errors->all() as $error) {
                $errorMessages[] = $error;
            }
            
            notify()->error('Validation Errors: ' . implode(', ', $errorMessages), 'Error');
            
            // Log all errors for debugging
            \Log::error('Schema Update Validation Errors:', $errors->toArray());

            return redirect()->back();
        }

        $schema = Schema::find($id);
        $input = $request->all();

        $finalData = [
            'name' => $input['name'],

            'type' => $input['type'],
            'min_amount' => $input['type'] == 'fixed' ? 0 : $input['min_amount'],
            'max_amount' => $input['type'] == 'fixed' ? 0 : $input['max_amount'],
            'fixed_amount' => $input['type'] == 'range' ? 0 : $input['fixed_amount'],

            'capital_back' => $input['capital_back'],

            'featured' => $input['featured'],
            'badge' => $input['featured'] == 1 ? $input['badge'] : null,

            'status' => $input['status'],
            'return_interest' => $input['return_interest'],
            'interest_type' => $input['interest_type'],
            'return_period' => $input['return_period'],
            'return_type' => $input['return_type'],
            'number_of_period' => $input['return_type'] == 'period' ? $input['number_of_period'] : 0,
            'off_days' => isset($input['off_days']) ? json_encode($input['off_days']) : null,
            'schema_duration' => $input['schema_duration'] ?? null,
            'schema_cancel' => $input['schema_cancel'],
            'expiry_minute' => $input['schema_cancel'] != 0 ? $input['expiry_minute'] : $schema->expiry_minute,
            'early_withdrawal_deduction' => $input['early_withdrawal_deduction'] ?? $schema->early_withdrawal_deduction ?? 25,
            'money_back_period' => $input['money_back_period'] ?? $schema->money_back_period ?? 15,
            'is_trending' => $input['is_trending'],
            'icon' => $request->hasFile('icon') ? self::imageUploadTrait($input['icon']) : $schema->icon,
        ];

        $schema->update($finalData);

        notify()->success('schema Update successfully');

        return redirect()->route('admin.schema.index');
    }
}
