<?php

namespace App\Http\Controllers;

use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Models\Invest;
use App\Models\LevelReferral;
use App\Models\Ranking;
use App\Models\Referral;
use App\Models\ReferralLink;
use App\Models\ReferralRelationship;
use App\Models\Schema;
use App\Models\Transaction;
use App\Models\User;
use App\Traits\NotifyTrait;
use Artisan;
use Carbon\Carbon;
use Txn;

class CronJobController extends Controller
{
    use NotifyTrait;

    /**
     * @return string
     */
    public function investmentCronJob()
    {

        $ongoingInvestment = Invest::where('status', InvestStatus::Ongoing)->where('next_profit_time', '<=', Carbon::now('Asia/Karachi')->format('Y-m-d H:i:s'))->cursor();

        foreach ($ongoingInvestment as $invest) {
            $schema = Schema::find($invest->schema_id);

            $profitOffDays = json_decode($schema->off_days, true);
            $date = Carbon::now('Asia/Karachi');
            $today = $date->format('l');

            if ($profitOffDays == null || !in_array($today, $profitOffDays)) {

                $user = User::find($invest->user_id);
                $calculateInterest = ($invest->interest * $invest->invest_amount) / 100;
                $interest = $invest->interest_type != 'percentage' ? $invest->interest : $calculateInterest;

                // Determine next profit time anchored to investment start
                $rawTime = optional($schema->schedule)->time
                    ?? ($invest->period_hours ?: 24);
                $scheduleMinutes = $rawTime > 48 ? $rawTime : ($rawTime * 60);
                $anchorStart = Carbon::parse($invest->created_at, 'Asia/Karachi');
                $elapsedMinutes = $anchorStart->diffInMinutes(Carbon::now('Asia/Karachi'));
                $cyclesElapsed = intdiv(max(0, $elapsedMinutes), max(1, $scheduleMinutes));
                $nextProfitTime = $anchorStart->copy()->addMinutes(($cyclesElapsed + 1) * $scheduleMinutes);

                $updateData = [
                    'next_profit_time' => $nextProfitTime,
                    'last_profit_time' => Carbon::now('Asia/Karachi'),
                    'number_of_period' => ($invest->number_of_period - 1),
                    'already_return_profit' => ($invest->already_return_profit + 1),
                    'total_profit_amount' => ($invest->total_profit_amount + $interest),
                ];

                $shortcodes = [
                    '[[full_name]]' => $user->full_name,
                    '[[plan_name]]' => $schema->name,
                    '[[invest_amount]]' => $invest->invest_amount,
                    '[[roi]]' => $interest,
                    '[[site_title]]' => setting('site_title', 'global'),
                    '[[site_url]]' => route('home'),
                ];

                if ($invest->return_type == 'lifetime') {
                    $invest->update($updateData);

                    $user->increment('profit_balance', $interest);
                    $tnxInfo = Txn::new($interest, 0, $interest, 'system', $schema->name . ' Plan Interest', TxnType::Interest, TxnStatus::Success, null, null, $user->id);

                    if (setting('site_referral', 'global') == 'level' && setting('profit_level')) {
                        $level = LevelReferral::where('type', 'profit')->max('the_order') + 1;
                        creditReferralBonus($tnxInfo->user, 'profit', $interest, $level);
                    }

                    //Notify newsletter
                    $this->mailNotify($tnxInfo->user->email, 'invest_roi', array_merge($shortcodes, ['[[txn]]' => $tnxInfo->tnx]));
                    $this->smsNotify('invest_roi', array_merge($shortcodes, ['[[txn]]' => $tnxInfo->tnx]), $tnxInfo->user->phone);
                    $this->pushNotify('invest_roi', $shortcodes, route('user.transactions'), $tnxInfo->user->id);

                } else {

                    if ($invest->number_of_period > 0) {
                        if ($invest->number_of_period == 1) {
                            $updateData = array_merge($updateData, [
                                'status' => InvestStatus::Completed,
                            ]);

                            if ($invest->capital_back == 1) {
                                $user->increment('balance', $invest->invest_amount);
                                $tnxInfo = Txn::new($invest->invest_amount, 0, $invest->invest_amount, 'system', $schema->name . ' Capital Back', TxnType::Refund, TxnStatus::Success, null, null, $user->id);

                                //notify newsletter
                                $this->mailNotify($tnxInfo->user->email, 'investment_end', array_merge($shortcodes, ['[[roi]]' => '','[[txn]]' => $tnxInfo->tnx]));
                                $this->smsNotify('investment_end', array_merge($shortcodes, ['[[roi]]' => '','[[txn]]' => $tnxInfo->tnx]), $tnxInfo->user->phone);
                                $this->pushNotify('investment_end', array_merge($shortcodes, ['[[roi]]' => '','[[txn]]' => $tnxInfo->tnx]), route('user.transactions'), $tnxInfo->user->id);
                            }

                        }

                        $invest->update($updateData);

                        $user->increment('profit_balance', $interest);
                        $tnxInfo = Txn::new($interest, 0, $interest, 'system', $schema->name . ' Plan Interest', TxnType::Interest, TxnStatus::Success, null, null, $user->id);

                        if (setting('site_referral', 'global') == 'level' && setting('profit_level')) {
                            $level = LevelReferral::where('type', 'profit')->max('the_order') + 1;
                            creditReferralBonus($tnxInfo->user, 'profit', $interest, $level);
                        }

                        //Notify newsletter
                        $this->mailNotify($tnxInfo->user->email, 'invest_roi', array_merge($shortcodes, ['[[txn]]' => $tnxInfo->tnx]));
                        $this->smsNotify('invest_roi', array_merge($shortcodes, ['[[txn]]' => $tnxInfo->tnx]), $tnxInfo->user->phone);
                        $this->pushNotify('invest_roi', array_merge($shortcodes, ['[[txn]]' => $tnxInfo->tnx]), route('user.transactions'), $tnxInfo->user->id);
                    }

                }
            }
        }

        return '....cron job successfully completed';
    }

    /**
     * @return string
     */
    public function referralCronJob()
    {
        if (setting('site_referral', 'global') == 'level') {
            return '....';
        }
        $referrals = Referral::all();
        $referralRelationship = ReferralRelationship::all();

        foreach ($referralRelationship as $relationship) {
            $provider = ReferralLink::find($relationship->referral_link_id)->user;

            $user = User::find($relationship->user_id);

            $totalDeposit = $user->totalDeposit();
            $totalInvest = $user->totalInvestment();

            $filterReferrals = $referrals->reject(function ($referral) use ($provider, $user) {

                return Transaction::where(function ($query) use ($user, $provider, $referral) {
                    $query->where('target_id', '!=', null)
                        ->where('user_id', $provider->id)
                        ->where('from_user_id', $user->id)
                        ->where('target_id', $referral->referral_target_id)
                        ->where('target_type', $referral->type)
                        ->where('is_level', 0);

                })->exists();

            })->map(function ($referral) {
                return $referral;
            });

            foreach ($filterReferrals as $referral) {

                $referralBonus = ($referral->bounty * $referral->target_amount) / 100;

                $targetName = $referral->target->name;

                if ($referral->type == 'deposit' && $referral->target_amount <= $totalDeposit && setting('deposit_referral_bounty', 'permission')) {
                    Txn::new($referralBonus, 0, $referralBonus, 'system', 'Referral Bonus with ' . $targetName . ' Via ' . $user->full_name, TxnType::Referral, TxnStatus::Success, null, null, $provider->id, $user->id, 'User', [], 'none', $referral->referral_target_id, $referral->type);
                    $provider->increment('profit_balance', $referralBonus);
                }

                if ($referral->type == 'investment' && $referral->target_amount <= $totalInvest && setting('investment_referral_bounty', 'permission')) {
                    Txn::new($referralBonus, 0, $referralBonus, 'system', 'Referral Bonus with ' . $targetName . ' Via ' . $user->full_name, TxnType::Referral, TxnStatus::Success, null, null, $provider->id, $user->id, 'User', [], 'none', $referral->referral_target_id, $referral->type);
                    $provider->increment('profit_balance', $referralBonus);
                }

            }

        }

        return '....referral job successfully completed';

    }

    /**
     * @return string
     */
    public function userRanking()
    {

        // Respect custom progression order using `ranking` sequence (not raw id)
        $rankings = Ranking::where('status', '=', true)->orderBy('ranking', 'asc')->get();

		foreach (User::where('status', true)->get() as $user) {
			$eligibleRanks = $rankings->reject(function ($rank) use ($user) {
				// User's own totals
				$totalEarning = $user->totalProfit();
				$totalDeposit = $user->totalDeposit();
				$totalInvest = $user->totalInvestment();

				// Direct referrals only (level 1)
				$directReferrals = $user->referrals;
				$directReferralCount = $directReferrals->count();

				// Admin-configured per-referral required amounts
				$perRefDeposit = (int) ($rank->minimum_referral_deposit ?? 0);
				$perRefInvest = (int) ($rank->minimum_referral_invest ?? 0);
				$requiredRefCount = (int) ($rank->minimum_referral ?? 0);

				// Count how many direct referrals individually meet per-ref requirements
				$qualifyingReferralsCount = $directReferrals->filter(function ($ref) use ($perRefDeposit, $perRefInvest) {
					$refTotalDeposit = (int) ($ref->total_deposit ?? 0);
					$refTotalInvest = (int) ($ref->total_invest ?? 0);
					return ($perRefDeposit <= 0 || $refTotalDeposit >= $perRefDeposit)
						&& ($perRefInvest <= 0 || $refTotalInvest >= $perRefInvest);
				})->count();

				// Cap each referral's contribution to the per-ref requirement before summing team totals
				$teamReferralDepositCapped = $directReferrals->sum(function ($ref) use ($perRefDeposit) {
					$val = (int) ($ref->total_deposit ?? 0);
					return $perRefDeposit > 0 ? min($val, $perRefDeposit) : $val;
				});
				$teamReferralInvestCapped = $directReferrals->sum(function ($ref) use ($perRefInvest) {
					$val = (int) ($ref->total_invest ?? 0);
					return $perRefInvest > 0 ? min($val, $perRefInvest) : $val;
				});

				// Required team totals based on required count and per-ref caps
				$requiredTeamDeposit = $perRefDeposit > 0 && $requiredRefCount > 0 ? ($perRefDeposit * $requiredRefCount) : 0;
				$requiredTeamInvest = $perRefInvest > 0 && $requiredRefCount > 0 ? ($perRefInvest * $requiredRefCount) : 0;

				return in_array($rank->id, json_decode($user->rankings)) ||
					// Own requirements
					$rank->minimum_earnings > $totalEarning ||
					$rank->minimum_deposit > $totalDeposit ||
					$rank->minimum_invest > $totalInvest ||
					// Direct referral count requirement
					($requiredRefCount > 0 && $directReferralCount < $requiredRefCount) ||
					// At least N qualifying referrals (meeting both per-ref thresholds)
					($requiredRefCount > 0 && $qualifyingReferralsCount < $requiredRefCount) ||
					// Team totals with per-ref caps must reach required totals
					($requiredTeamDeposit > 0 && $teamReferralDepositCapped < $requiredTeamDeposit) ||
					($requiredTeamInvest > 0 && $teamReferralInvestCapped < $requiredTeamInvest);
			});

            // Promote only one step at a time to the next sequence rank if eligible
            if ($eligibleRanks->isNotEmpty()) {
                $currentSequence = optional($user->rank)->ranking ?? 1;
                $nextEligible = $eligibleRanks
                    ->filter(function ($rank) use ($currentSequence) { return $rank->ranking > $currentSequence; })
                    ->sortBy('ranking')
                    ->first();

                if ($nextEligible) {
                    // Credit bonus once at promotion
                    if ($nextEligible->bonus > 0) {
                        Txn::new($nextEligible->bonus, 0, $nextEligible->bonus, 'system', 'Ranking Bonus: ' . $nextEligible->ranking_name, TxnType::Bonus, TxnStatus::Success, null, null, $user->id);
                        $user->profit_balance += $nextEligible->bonus;
                    }

                    $completed = json_decode($user->rankings) ?: [];
                    if (!in_array($nextEligible->id, $completed)) {
                        $completed[] = $nextEligible->id;
                    }

                    $user->update([
                        'ranking_id' => $nextEligible->id,
                        'rankings' => json_encode($completed),
                    ]);
                }
            }
        }

        return '....referral job successfully completed';

    }

    public function queueWork()
    {
        Artisan::call('queue:work');

        return '.... Running successfully';
    }
}