<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\GatewayType;
use App\Http\Controllers\Controller;
use App\Models\DepositMethod;
use App\Traits\NotifyTrait;
use App\Traits\Payment;

class GatewayController extends Controller
{
    use NotifyTrait, Payment;

    public function gateway($code)
    {
        try {
            \Log::info('Gateway request for code:', ['code' => $code]);
            
            // Log all available gateway codes
            $allGateways = DepositMethod::where('status', 1)->get(['id', 'name', 'gateway_code']);
            \Log::info('All available gateways:', ['gateways' => $allGateways->toArray()]);
            
            $gateway = DepositMethod::code($code)->first();
            
            \Log::info('Gateway found:', ['gateway' => $gateway ? $gateway->toArray() : 'null']);
            \Log::info('Payment details:', ['payment_details' => $gateway ? $gateway->payment_details : 'null']);
            
            if (!$gateway) {
                \Log::error('Gateway not found for code:', ['code' => $code]);
                return response()->json(['error' => 'Gateway not found'], 404);
            }
            
            // Prepare payment details based on gateway type
            $paymentDetails = $gateway->payment_details ?? '{}';
            $fieldOptions = $gateway->field_options ?? '[]';

            // For bank-style gateways, keep only bank related keys if present
            if (strpos(strtolower($gateway->name), 'bank') !== false ||
                strpos(strtolower($gateway->name), 'jazzcash') !== false ||
                strpos(strtolower($gateway->name), 'easypaisa') !== false) {
                $paymentDetailsArray = json_decode($paymentDetails, true);
                if (is_array($paymentDetailsArray)) {
                    // Do not drop information; just pass through as provided by admin
                    $paymentDetails = json_encode($paymentDetailsArray);
                }
            }

            // Generate credentials HTML for any gateway that has payment details configured.
            // Previously this only rendered when type === 'manual' AND fieldOptions existed, which hid bank details.
            $credentials = '';
            if (!empty($paymentDetails) && $paymentDetails !== '{}') {
                $credentials = view('frontend.shahdeveloper.gateway.include.manual', [
                    'paymentDetails' => $paymentDetails,
                    'fieldOptions' => $fieldOptions ?: '[]',
                ])->render();
            }
            
            $response = [
                'id' => $gateway->id,
                'name' => $gateway->name,
                'gateway_code' => $gateway->gateway_code,
                'type' => $gateway->type,
                'charge' => $gateway->charge,
                'charge_type' => $gateway->charge_type,
                'minimum_deposit' => $gateway->minimum_deposit,
                'maximum_deposit' => $gateway->maximum_deposit,
                'currency' => $gateway->currency ?? 'USD',
                'currency_symbol' => $gateway->currency_symbol ?? '$',
                'rate' => $gateway->rate ?? 1,
                'payment_details' => $paymentDetails,
                'charge_gateway' => $gateway->charge . ($gateway->charge_type === 'percentage' ? ' %' : ' ' . 'USD'),
                'credentials' => $credentials
            ];

            return response()->json($response);
            
        } catch (\Exception $e) {
            \Log::error('Gateway Error: ' . $e->getMessage());
            return response()->json(['error' => 'Server error'], 500);
        }
    }

    //list json
    public function gatewayList()
    {
        $gateways = DepositMethod::where('status', 1)->get();

        return view('frontend::gateway.include.__list', compact('gateways'));
    }

    // Get gateway details by ID
    public function gatewayDetails($id)
    {
        try {
            \Log::info('Gateway details request for ID:', ['id' => $id]);
            
            $gateway = DepositMethod::where('id', $id)->where('status', 1)->first();
            
            if (!$gateway) {
                \Log::error('Gateway not found for ID:', ['id' => $id]);
                return response()->json(['error' => 'Gateway not found'], 404);
            }
            
            \Log::info('Gateway found:', ['gateway' => $gateway->toArray()]);
            
            $response = [
                'id' => $gateway->id,
                'name' => $gateway->name,
                'gateway_code' => $gateway->gateway_code,
                'type' => $gateway->type,
                'payment_details' => $gateway->payment_details,
            ];

            return response()->json($response);
            
        } catch (\Exception $e) {
            \Log::error('Gateway Details Error: ' . $e->getMessage());
            return response()->json(['error' => 'Server error'], 500);
        }
    }
}
