<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Models\DepositMethod;
use App\Models\Invest;
use App\Models\LevelReferral;
use App\Models\Schema;
use App\Traits\ImageUpload;
use App\Traits\NotifyTrait;
use Auth;
use Carbon\Carbon;
use DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Txn;

class ImprovedInvestController extends GatewayController
{
    use ImageUpload, NotifyTrait;

    public function investNow(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'schema_id' => 'required|exists:schemas,id',
            'invest_amount' => 'required|numeric|min:0.01',
            'wallet' => 'required|in:main,profit,gateway',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        try {
            DB::beginTransaction();
            
            $result = $this->processInvestment($request->all());
            
            DB::commit();
            
            notify()->success('Investment created successfully', 'Success');
            return redirect()->route('user.collect-roi');
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Investment creation failed', [
                'user_id' => auth()->id(),
                'request' => $request->all(),
                'error' => $e->getMessage()
            ]);
            
            notify()->error($e->getMessage(), 'Error');
            return redirect()->back();
        }
    }

    private function processInvestment(array $input)
    {
        $user = Auth::user();
        $schema = Schema::with('schedule')->findOrFail($input['schema_id']);
        $investAmount = (float) $input['invest_amount'];

        // Validate investment amount
        $this->validateInvestmentAmount($schema, $investAmount);
        
        // Validate wallet balance
        $this->validateWalletBalance($user, $input['wallet'], $investAmount);

        // Calculate next profit time using minutes schedule
        $scheduleMinutes = optional($schema->schedule)->time ?? 1440; // Default 24 hours in minutes
        $nextProfitTime = Carbon::now()->addMinutes($scheduleMinutes);

        $investmentData = [
            'user_id' => $user->id,
            'schema_id' => $schema->id,
            'invest_amount' => $investAmount,
            'next_profit_time' => $nextProfitTime,
            'capital_back' => $schema->capital_back,
            'interest' => $schema->return_interest,
            'interest_type' => $schema->interest_type,
            'return_type' => $schema->return_type,
            'number_of_period' => $schema->number_of_period,
            'period_hours' => round($scheduleMinutes / 60, 1), // Convert minutes to hours for display
            'wallet' => $input['wallet'],
            'status' => InvestStatus::Ongoing,
            'already_return_profit' => 0,
            'created_at' => Carbon::now(),
        ];

        if ($input['wallet'] === 'gateway') {
            return $this->processGatewayPayment($input, $investmentData, $schema);
        }

        // Process wallet payment
        $this->deductFromWallet($user, $input['wallet'], $investAmount);
        
        // Create transaction
        $txnInfo = Txn::new(
            $investAmount,
            0,
            $investAmount,
            'system',
            $schema->name . ' Investment',
            TxnType::Investment,
            TxnStatus::Success,
            null,
            null,
            $user->id
        );

        $investmentData['transaction_id'] = $txnInfo->id;
        $investment = Invest::create($investmentData);

        // Process referral bonus
        $this->processReferralBonus($user, $investAmount);
        
        // Send notifications
        $this->sendInvestmentNotifications($txnInfo, $schema);

        return $investment;
    }

    private function validateInvestmentAmount(Schema $schema, float $amount)
    {
        if ($schema->type === 'range') {
            if ($amount < $schema->min_amount || $amount > $schema->max_amount) {
                throw new \Exception("Investment amount must be between {$schema->min_amount} and {$schema->max_amount}");
            }
        } elseif ($schema->type === 'fixed') {
            if ($amount != $schema->fixed_amount) {
                throw new \Exception("Investment amount must be exactly {$schema->fixed_amount}");
            }
        }
    }

    private function validateWalletBalance(User $user, string $wallet, float $amount)
    {
        $balance = $wallet === 'main' ? $user->balance : $user->profit_balance;
        
        if ($balance < $amount) {
            $walletName = $wallet === 'main' ? 'Main' : 'Profit';
            throw new \Exception("Insufficient balance in your {$walletName} wallet");
        }
    }

    private function deductFromWallet(User $user, string $wallet, float $amount)
    {
        if ($wallet === 'main') {
            $user->decrement('balance', $amount);
        } else {
            $user->decrement('profit_balance', $amount);
        }
    }

    private function processReferralBonus(User $user, float $amount)
    {
        // REMOVED: Investment referral commission disabled
        // Only deposit referral commission is allowed (one-time)
    }

    private function sendInvestmentNotifications($txnInfo, $schema)
    {
        $shortcodes = [
            '[[full_name]]' => $txnInfo->user->full_name,
            '[[txn]]' => $txnInfo->tnx,
            '[[plan_name]]' => $schema->name,
            '[[invest_amount]]' => $txnInfo->amount . setting('site_currency', 'global'),
            '[[site_title]]' => setting('site_title', 'global'),
            '[[site_url]]' => route('home'),
        ];

        $this->mailNotify($txnInfo->user->email, 'user_investment', $shortcodes);
        $this->pushNotify('user_investment', $shortcodes, route('user.collect-roi'), $txnInfo->user->id);
        $this->smsNotify('user_investment', $shortcodes, $txnInfo->user->phone);
    }

    public function investLogs(Request $request)
    {
        $data = Invest::with(['schema', 'user'])
            ->where('user_id', auth()->id())
            ->latest();

        if ($request->ajax()) {
            return $this->getInvestmentDataTable($data);
        }

        return view('frontend::user.invest.improved-log', compact('data'));
    }

    private function getInvestmentDataTable($data)
    {
        return Datatables::of($data)
            ->addIndexColumn()
            ->addColumn('schema_info', function ($invest) {
                return [
                    'name' => $invest->schema->name,
                    'amount' => $invest->invest_amount,
                    'created_at' => $invest->created_at->format('Y-m-d H:i:s')
                ];
            })
            ->addColumn('roi', function ($invest) {
                return $invest->interest_type == 'percentage' 
                    ? $invest->interest . '%' 
                    : setting('site_currency', 'global') . $invest->interest;
            })
            ->addColumn('profit_info', function ($invest) {
                $profitAmount = $invest->interest_type == 'percentage' 
                    ? ($invest->interest * $invest->invest_amount) / 100
                    : $invest->interest;
                
                return [
                    'per_period' => $profitAmount,
                    'total_earned' => $invest->already_return_profit * $profitAmount,
                    'periods_completed' => $invest->already_return_profit
                ];
            })
            ->addColumn('status_info', function ($invest) {
                return [
                    'status' => $invest->status->value,
                    'next_profit_time' => $invest->next_profit_time,
                    'periods_remaining' => $invest->return_type == 'period' 
                        ? max(0, $invest->number_of_period - $invest->already_return_profit)
                        : 'Unlimited'
                ];
            })
            ->make(true);
    }

    public function getInvestmentTimer($id)
    {
        $investment = Invest::findOrFail($id);
        
        if ($investment->user_id !== auth()->id()) {
            abort(403);
        }

        $now = Carbon::now();
        $nextProfitTime = Carbon::parse($investment->next_profit_time);
        $lastProfitTime = Carbon::parse($investment->last_profit_time ?? $investment->created_at);
        
        $totalDuration = $nextProfitTime->diffInSeconds($lastProfitTime);
        $elapsed = $now->diffInSeconds($lastProfitTime);
        $remaining = max(0, $nextProfitTime->diffInSeconds($now));
        
        $progress = $totalDuration > 0 ? min(100, ($elapsed / $totalDuration) * 100) : 100;

        return response()->json([
            'remaining_seconds' => $remaining,
            'progress_percentage' => round($progress, 2),
            'next_profit_time' => $nextProfitTime->toISOString(),
            'status' => $investment->status->value
        ]);
    }
}
