<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProfitDistribution extends Model
{
    use HasFactory;

    protected $fillable = [
        'investment_id',
        'user_id',
        'profit_amount',
        'distributed_by',
        'distribution_time',
        'notes'
    ];

    protected $casts = [
        'distribution_time' => 'datetime',
        'profit_amount' => 'decimal:2'
    ];

    /**
     * Get the investment that this distribution belongs to
     */
    public function investment()
    {
        return $this->belongsTo(Invest::class, 'investment_id');
    }

    /**
     * Get the user who received the profit
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the admin who distributed the profit
     */
    public function distributedBy()
    {
        return $this->belongsTo(Admin::class, 'distributed_by');
    }

    /**
     * Scope to get distributions by date range
     */
    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('distribution_time', [$startDate, $endDate]);
    }

    /**
     * Scope to get distributions by user
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope to get distributions by admin
     */
    public function scopeByAdmin($query, $adminId)
    {
        return $query->where('distributed_by', $adminId);
    }
}
