<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class RoiDistribution extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'investment_id',
        'roi_amount',
        'distribution_type', // 'manual', 'bulk', 'auto'
        'distributed_by',
        'distribution_time',
        'notes',
        'is_reversed',
        'reversed_at',
        'reversed_by',
        'reversal_reason'
    ];

    protected $casts = [
        'distribution_time' => 'datetime',
        'reversed_at' => 'datetime',
        'roi_amount' => 'decimal:2',
        'is_reversed' => 'boolean'
    ];

    /**
     * Get the user who received the ROI
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the investment that this ROI distribution belongs to
     */
    public function investment()
    {
        return $this->belongsTo(Invest::class);
    }

    /**
     * Get the admin who distributed the ROI
     */
    public function distributedBy()
    {
        return $this->belongsTo(Admin::class, 'distributed_by');
    }

    /**
     * Get the admin who reversed the ROI
     */
    public function reversedBy()
    {
        return $this->belongsTo(Admin::class, 'reversed_by');
    }

    /**
     * Scope to get active (non-reversed) distributions
     */
    public function scopeActive($query)
    {
        return $query->where('is_reversed', false);
    }

    /**
     * Scope to get reversed distributions
     */
    public function scopeReversed($query)
    {
        return $query->where('is_reversed', true);
    }

    /**
     * Scope to get distributions by date range
     */
    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('distribution_time', [$startDate, $endDate]);
    }

    /**
     * Scope to get distributions by user
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope to get distributions by admin
     */
    public function scopeByAdmin($query, $adminId)
    {
        return $query->where('distributed_by', $adminId);
    }

    /**
     * Scope to get distributions by type
     */
    public function scopeByType($query, $type)
    {
        return $query->where('distribution_type', $type);
    }

    /**
     * Get the transaction associated with this distribution
     */
    public function transaction()
    {
        return $this->hasOne(Transaction::class, 'roi_distribution_id');
    }

    /**
     * Check if this distribution can be reversed
     */
    public function canBeReversed()
    {
        // Can only reverse if not already reversed and within 24 hours
        if ($this->is_reversed) {
            return false;
        }

        $hoursSinceDistribution = Carbon::now()->diffInHours($this->distribution_time);
        return $hoursSinceDistribution <= 24;
    }

    /**
     * Get the time remaining to reverse this distribution
     */
    public function getTimeToReverseAttribute()
    {
        if ($this->is_reversed) {
            return 'Already reversed';
        }

        $hoursSinceDistribution = Carbon::now()->diffInHours($this->distribution_time);
        $hoursRemaining = 24 - $hoursSinceDistribution;

        if ($hoursRemaining <= 0) {
            return 'Expired';
        }

        return $hoursRemaining . ' hours';
    }
}
