<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class RoiSchedule extends Model
{
    use HasFactory;

    protected $fillable = [
        'invest_id',
        'user_id',
        'schema_id',
        'invest_amount',
        'roi_percentage',
        'roi_amount',
        'roi_start_time',
        'roi_end_time',
        'roi_ready_time',
        'roi_sent_time',
        'status',
        'roi_cycle',
        'total_cycles'
    ];

    protected $casts = [
        'roi_start_time' => 'datetime',
        'roi_end_time' => 'datetime',
        'roi_ready_time' => 'datetime',
        'roi_sent_time' => 'datetime',
        'invest_amount' => 'decimal:2',
        'roi_percentage' => 'decimal:2',
        'roi_amount' => 'decimal:2',
    ];

    // Relationships
    public function invest()
    {
        return $this->belongsTo(Invest::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function schema()
    {
        return $this->belongsTo(Schema::class);
    }

    // Helper methods
    public function isReady()
    {
        return $this->status === 'ready' && $this->roi_ready_time && $this->roi_ready_time <= now();
    }

    public function isPending()
    {
        return $this->status === 'pending';
    }

    public function isSent()
    {
        return $this->status === 'sent';
    }

    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    public function getTimeRemaining()
    {
        if ($this->status === 'pending') {
            $remaining = $this->roi_end_time->diffInSeconds(now());
            return max(0, $remaining);
        }
        return 0;
    }

    public function getTimeElapsed()
    {
        if ($this->status === 'pending') {
            return $this->roi_start_time->diffInSeconds(now());
        }
        return $this->roi_start_time->diffInSeconds($this->roi_end_time);
    }

    public function getProgressPercentage()
    {
        if ($this->status === 'ready' || $this->status === 'sent') {
            return 100; // If ready or sent, show 100% progress
        }
        
        $total = $this->roi_start_time->diffInSeconds($this->roi_end_time);
        $elapsed = $this->getTimeElapsed();
        return min(100, ($elapsed / $total) * 100);
    }

    public function markAsReady()
    {
        $this->update([
            'status' => 'ready',
            'roi_ready_time' => now()
        ]);
    }

    public function markAsSent()
    {
        $this->update([
            'status' => 'sent',
            'roi_sent_time' => now()
        ]);
    }

    public function createNextCycle()
    {
        if ($this->roi_cycle < $this->total_cycles) {
            // Use schema-configured period (in hours) -> convert to minutes
            $roiPeriodHours = optional($this->schema)->roi_period ?? 24; // default 24 hours
            $roiPeriodMinutes = (int)$roiPeriodHours * 60;

            // Next cycle starts from collection time (now), ends after configured period
            $nextStart = now();
            $nextEnd = now()->copy()->addMinutes($roiPeriodMinutes);

            return self::create([
                'invest_id' => $this->invest_id,
                'user_id' => $this->user_id,
                'schema_id' => $this->schema_id,
                'invest_amount' => $this->invest_amount,
                'roi_percentage' => $this->roi_percentage,
                'roi_amount' => $this->roi_amount,
                'roi_start_time' => $nextStart,
                'roi_end_time' => $nextEnd,
                'status' => 'pending',
                'roi_cycle' => $this->roi_cycle + 1,
                'total_cycles' => $this->total_cycles
            ]);
        }
        return null;
    }
}
