<?php

namespace App\Models;

use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Carbon\Carbon;
use Coderflex\LaravelTicket\Concerns\HasTickets;
use Coderflex\LaravelTicket\Contracts\CanUseTickets;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Facades\DB;

class User extends Authenticatable implements CanUseTickets, MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable, HasTickets;

    protected $fillable = [
        'ranking_id', 'rankings', 'avatar', 'first_name', 'last_name', 'country', 'phone',
        'username', 'email', 'email_verified_at', 'gender', 'date_of_birth', 'city',
        'zip_code', 'address', 'balance', 'profit_balance', 'status', 'kyc', 'kyc_credential',
        'google2fa_secret', 'two_fa', 'deposit_status', 'withdraw_status', 'transfer_status',
        'ref_id', 'password', 'last_salary_date',
    ];

    protected $appends = [
        'full_name', 'kyc_time', 'kyc_type', 'total_profit', 'total_deposit', 'total_invest', 'profile_completion_percentage',
    ];

    protected $dates = ['kyc_time'];

    protected $hidden = [
        'password', 'remember_token', 'google2fa_secret',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'two_fa' => 'boolean',
    ];

    public function ranking()
    {
        return $this->belongsTo(\App\Models\Ranking::class, 'ranking_id');
    }

    public function getAvatarAttribute()
    {
        return $this->attributes['avatar'] ?? 'frontend/images/user.png';
    }

    public function getUpdatedAtAttribute(): string
    {
        return Carbon::parse($this->attributes['updated_at'])->format('M d Y h:i');
    }

    public function getFullNameAttribute(): string
    {
        return ucwords(trim("{$this->first_name} {$this->last_name}"));
    }

    public function getKycTypeAttribute(): string
    {
        return json_decode($this->kyc_credential, true)['kyc_type_of_name'] ?? '';
    }

    public function getKycTimeAttribute(): string
    {
        return json_decode($this->kyc_credential, true)['kyc_time_of_time'] ?? '';
    }

    public function transaction()
    {
        return $this->hasMany(Transaction::class, 'user_id');
    }

    public function invests()
    {
        return $this->hasMany(Invest::class, 'user_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'user_id');
    }

    public function getTotalProfitAttribute(): string
    {
        return $this->totalProfit();
    }

    public function totalProfit($days = null)
    {
        $query = $this->transaction()
            ->where('status', TxnStatus::Success)
            ->whereIn('type', [TxnType::Referral, TxnType::SignupBonus, TxnType::Interest, TxnType::Bonus]);

        if ($days) {
            $query->where('created_at', '>=', Carbon::now()->subDays((int)$days));
        }

        return round($query->sum('amount'), 2);
    }

    public function getTotalDepositAttribute(): string
    {
        return $this->totalDeposit();
    }

    public function totalDeposit()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->whereIn('type', [TxnType::Deposit, TxnType::ManualDeposit])
            ->sum('amount'), 2);
    }

    public function getTotalInvestAttribute(): string
    {
        return $this->totalInvestment();
    }

    public function totalInvestment()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Investment)
            ->sum('amount'), 2);
    }

    public function totalRoiProfit()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Interest)
            ->sum('amount'), 2);
    }

    public function totalDepositBonus()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Referral)
            ->whereNotNull('target_id')
            ->where('target_type', 'deposit')
            ->sum('amount'), 2);
    }

    public function totalInvestBonus()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Referral)
            ->whereNotNull('target_id')
            ->where('target_type', 'investment')
            ->sum('amount'), 2);
    }

    public function totalWithdraw()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->whereIn('type', [TxnType::Withdraw, TxnType::WithdrawAuto])
            ->sum('amount'), 2);
    }

    public function totalTransfer()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::SendMoney)
            ->sum('amount'), 2);
    }

    public function totalReferralProfit()
    {
        return round($this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Referral)
            ->sum('amount'), 2);
    }

    public function getReferrals()
    {
        return ReferralProgram::all()->map(function ($program) {
            return ReferralLink::getReferral($this, $program);
        });
    }

    public function getReferralUsers()
    {
        return $this->referrals()
            ->withCount(['referrals as referrals_count'])
            ->select('users.*')
            ->addSelect([
                'balance' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type = "deposit")'),
                'pending_balance' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "pending" AND type = "deposit")'),
                'second_level_count' => DB::raw('(SELECT COUNT(*) FROM users as u2 WHERE u2.ref_id IN (SELECT id FROM users WHERE ref_id = users.id))'),
                'currentSale' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type IN ("deposit", "investment"))'),
                'requiredSale' => DB::raw('0'),
                'pendingAmount' => DB::raw('0'),
                'requirementMet' => DB::raw('false'),
                'referCount' => DB::raw('(SELECT COUNT(*) FROM users WHERE ref_id = users.id)'),
                'join_date' => DB::raw('users.created_at')
            ])
            ->get();
    }

    public function referrals()
    {
        return $this->hasMany(User::class, 'ref_id');
    }

    public function referrer()
    {
        return $this->belongsTo(User::class, 'ref_id');
    }

    // ✅ ADDED: Get total referral count
    public function getTotalReferralCount()
    {
        return $this->referrals()->count();
    }

    // ✅ ADDED: Get referral count by date range
    public function getReferralCountByDate($days = 30)
    {
        return $this->referrals()
            ->where('created_at', '>=', now()->subDays($days))
            ->count();
    }

    // ✅ ADDED: Get referral earnings
    public function getReferralEarnings($days = null)
    {
        $query = $this->transaction()
            ->where('status', TxnStatus::Success)
            ->where('type', TxnType::Referral);
        
        if ($days) {
            $query->where('created_at', '>=', now()->subDays($days));
        }
        
        return round($query->sum('amount'), 2);
    }

    // ✅ ADDED: Get total referral investments
    public function getTotalReferralInvestments()
    {
        return $this->referrals()
            ->join('invests', 'users.id', '=', 'invests.user_id')
            ->where('invests.status', 'ongoing')
            ->sum('invests.invest_amount');
    }

    // ✅ ADDED: Get total referral deposits
    public function getTotalReferralDeposits()
    {
        return $this->referrals()
            ->join('transactions', 'users.id', '=', 'transactions.user_id')
            ->where('transactions.status', 'success')
            ->where('transactions.type', 'deposit')
            ->sum('transactions.amount');
    }

    // ✅ ADDED: Get total referral profits
    public function getTotalReferralProfits()
    {
        return $this->referrals()
            ->join('transactions', 'users.id', '=', 'transactions.user_id')
            ->where('transactions.status', 'success')
            ->where('transactions.type', 'profit')
            ->sum('transactions.amount');
    }

    // ✅ ADDED: Get referral investment summary
    public function getReferralInvestmentSummary()
    {
        return [
            'total_investments' => $this->getTotalReferralInvestments(),
            'total_deposits' => $this->getTotalReferralDeposits(),
            'total_profits' => $this->getTotalReferralProfits(),
            'total_referrals' => $this->getTotalReferralCount()
        ];
    }

    public function rank()
    {
        return $this->belongsTo(Ranking::class, 'ranking_id');
    }

    public function ticket()
    {
        return $this->hasMany(Ticket::class);
    }

    public function rankAchieved()
    {
        return count(json_decode($this->rankings, true));
    }

    /**
     * Calculate profile completion percentage
     */
    public function getProfileCompletionPercentageAttribute()
    {
        $totalFields = 0;
        $completedFields = 0;

        // Core required fields (6 fields = 50% when all filled)
        $coreFields = [
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'username' => $this->username,
            'email' => $this->email,
            'phone' => $this->phone,
            'country' => $this->country,
        ];

        // Additional fields (6 fields = 50% when all filled)
        $additionalFields = [
            'city' => $this->city,
            'address' => $this->address,
            'zip_code' => $this->zip_code,
            'gender' => $this->gender,
            'date_of_birth' => $this->date_of_birth,
            'avatar' => $this->avatar,
        ];

        // Count core fields (50% weight)
        foreach ($coreFields as $field => $value) {
            $totalFields++;
            if (!empty($value)) {
                $completedFields++;
            }
        }

        // Count additional fields (50% weight)
        foreach ($additionalFields as $field => $value) {
            $totalFields++;
            if (!empty($value)) {
                $completedFields++;
            }
        }

        return $totalFields > 0 ? round(($completedFields / $totalFields) * 100) : 0;
    }

    /**
     * Check if profile is 80% complete
     */
    public function isProfileComplete()
    {
        return $this->profile_completion_percentage >= 80;
    }

    protected function google2faSecret(): Attribute
    {
        return new Attribute(
            get: fn($value) => $value ? decrypt($value) : null,
            set: fn($value) => encrypt($value),
        );
    }
}
