<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Event;
use Illuminate\Auth\Events\Registered;

class AvatarServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Listen for user registration event
        Event::listen(Registered::class, function ($event) {
            $this->generateAvatarForUser($event->user);
        });

        // Listen for user creation (for admin-created users)
        User::created(function ($user) {
            $this->generateAvatarForUser($user);
        });
    }

    private function generateAvatarForUser($user)
    {
        try {
            // Only generate if user doesn't have a custom avatar
            if (empty($user->avatar) || str_contains($user->avatar, 'frontend/images/user.png')) {
                $avatarPath = $this->generateAvatar($user->username, $user->id);
                
                // Update user's avatar
                $user->avatar = $avatarPath;
                $user->saveQuietly(); // Save without triggering events
            }
        } catch (\Exception $e) {
            // Log error but don't break user registration
            \Log::warning("Failed to generate avatar for user {$user->username}: " . $e->getMessage());
        }
    }

    private function generateAvatar($username, $userId)
    {
        // Create avatar directory if it doesn't exist
        $avatarDir = 'public/avatars';
        if (!Storage::exists($avatarDir)) {
            Storage::makeDirectory($avatarDir);
        }

        // Generate unique filename
        $filename = 'avatar_' . $userId . '_' . time() . '.svg';
        $filepath = $avatarDir . '/' . $filename;

        // Generate SVG avatar
        $svg = $this->generateSVGAvatar($username);
        
        // Save SVG file
        Storage::put($filepath, $svg);

        // Return the public URL - fix the path to match storage link
        return '/storage/avatars/' . $filename;
    }

    private function generateSVGAvatar($username)
    {
        // Get first letter of username
        $initial = strtoupper(substr($username, 0, 1));
        
        // Generate random colors
        $colors = [
            '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
            '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9',
            '#F8C471', '#82E0AA', '#F1948A', '#85C1E9', '#D7BDE2'
        ];
        $bgColor = $colors[array_rand($colors)];
        
        // Generate complementary text color
        $textColor = $this->getContrastColor($bgColor);

        $svg = '<?xml version="1.0" encoding="UTF-8"?>
        <svg width="100" height="100" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
            <defs>
                <linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="100%">
                    <stop offset="0%" style="stop-color:' . $bgColor . ';stop-opacity:1" />
                    <stop offset="100%" style="stop-color:' . $this->darkenColor($bgColor, 20) . ';stop-opacity:1" />
                </linearGradient>
            </defs>
            <circle cx="50" cy="50" r="50" fill="url(#grad)" />
            <text x="50" y="65" font-family="Arial, sans-serif" font-size="40" font-weight="bold" text-anchor="middle" fill="' . $textColor . '">' . $initial . '</text>
        </svg>';

        return $svg;
    }

    private function getContrastColor($hexColor)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Calculate luminance
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
        
        // Return black or white based on luminance
        return $luminance > 0.5 ? '#000000' : '#FFFFFF';
    }

    private function darkenColor($hexColor, $percent)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Darken by percentage
        $r = max(0, min(255, $r - ($r * $percent / 100)));
        $g = max(0, min(255, $g - ($g * $percent / 100)));
        $b = max(0, min(255, $b - ($b * $percent / 100)));
        
        // Convert back to hex
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }
}
