@extends('backend.layouts.app')

@section('title', 'ROI Distribution History')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="card-title">
                            <i class="fas fa-history"></i> ROI Distribution History
                        </h4>
                        <a href="{{ route('admin.roi-tracking.index') }}" class="btn btn-primary">
                            <i class="fas fa-arrow-left"></i> Back to ROI Tracking
                        </a>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Filters -->
                    <div class="row mb-3">
                        <div class="col-md-3">
                            <label>Start Date:</label>
                            <input type="date" class="form-control" id="startDate" value="{{ request('start_date', date('Y-m-d', strtotime('-30 days'))) }}">
                        </div>
                        <div class="col-md-3">
                            <label>End Date:</label>
                            <input type="date" class="form-control" id="endDate" value="{{ request('end_date', date('Y-m-d')) }}">
                        </div>
                        <div class="col-md-2">
                            <label>Status:</label>
                            <select class="form-control" id="statusFilter">
                                <option value="">All Status</option>
                                <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                <option value="reversed" {{ request('status') == 'reversed' ? 'selected' : '' }}>Reversed</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-primary btn-block" onclick="applyFilters()">
                                <i class="fas fa-filter"></i> Apply
                            </button>
                        </div>
                        <div class="col-md-2">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-secondary btn-block" onclick="resetFilters()">
                                <i class="fas fa-undo"></i> Reset
                            </button>
                        </div>
                    </div>

                    <!-- Statistics Cards -->
                    <div class="row mb-4">
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-info">
                                <div class="inner">
                                    <h3>{{ $distributions->total() }}</h3>
                                    <p>Total Distributions</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-chart-line"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-success">
                                <div class="inner">
                                    <h3>{{ $distributions->where('is_reversed', false)->count() }}</h3>
                                    <p>Active Distributions</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-check"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-warning">
                                <div class="inner">
                                    <h3>{{ $distributions->where('is_reversed', true)->count() }}</h3>
                                    <p>Reversed Distributions</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-undo"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-primary">
                                <div class="inner">
                                    <h3>${{ number_format($distributions->where('is_reversed', false)->sum('roi_amount'), 2) }}</h3>
                                    <p>Total Active ROI</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-dollar-sign"></i>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Distributions Table -->
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>User</th>
                                    <th>Investment</th>
                                    <th>ROI Amount</th>
                                    <th>Type</th>
                                    <th>Distributed By</th>
                                    <th>Distribution Time</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($distributions as $distribution)
                                <tr>
                                    <td>
                                        <code>#{{ $distribution->id }}</code>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <img src="{{ asset($distribution->user->avatar ?? 'assets/global/images/default-avatar.png') }}" 
                                                 class="rounded-circle mr-2" width="35" height="35">
                                            <div>
                                                <strong>{{ $distribution->user->full_name ?? 'Unknown User' }}</strong><br>
                                                <small class="text-muted">{{ $distribution->user->email ?? 'N/A' }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div>
                                            <strong>{{ $distribution->investment->schema->name ?? 'Unknown Plan' }}</strong><br>
                                            <small class="text-muted">
                                                ${{ number_format($distribution->investment->invest_amount ?? 0, 2) }}
                                                ({{ $distribution->investment->already_return_profit ?? 0 }}/{{ $distribution->investment->number_of_period ?? 0 }})
                                            </small>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="text-success font-weight-bold">
                                            ${{ number_format($distribution->roi_amount, 2) }}
                                        </span>
                                    </td>
                                    <td>
                                        @php
                                            $typeColors = [
                                                'manual' => 'primary',
                                                'bulk' => 'success',
                                                'auto' => 'info'
                                            ];
                                            $color = $typeColors[$distribution->distribution_type] ?? 'secondary';
                                        @endphp
                                        <span class="badge badge-{{ $color }}">
                                            {{ ucfirst($distribution->distribution_type) }}
                                        </span>
                                    </td>
                                    <td>
                                        <small>{{ $distribution->distributedBy->name ?? 'Unknown Admin' }}</small>
                                    </td>
                                    <td>
                                        <div>
                                            <small>{{ \Carbon\Carbon::parse($distribution->distribution_time)->format('M d, Y H:i') }}</small><br>
                                            <small class="text-muted">{{ \Carbon\Carbon::parse($distribution->distribution_time)->diffForHumans() }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        @if($distribution->is_reversed)
                                            <div>
                                                <span class="badge badge-danger">Reversed</span><br>
                                                <small class="text-muted">
                                                    {{ \Carbon\Carbon::parse($distribution->reversed_at)->format('M d, Y H:i') }}
                                                </small>
                                            </div>
                                        @else
                                            <div>
                                                <span class="badge badge-success">Active</span><br>
                                                <small class="text-muted">{{ $distribution->time_to_reverse }}</small>
                                            </div>
                                        @endif
                                    </td>
                                    <td>
                                        @if(!$distribution->is_reversed && $distribution->canBeReversed())
                                            <button type="button" class="btn btn-sm btn-warning" 
                                                    onclick="showReverseModal({{ $distribution->id }}, '{{ $distribution->user->full_name ?? 'Unknown User' }}', {{ $distribution->roi_amount }})">
                                                <i class="fas fa-undo"></i> Reverse
                                            </button>
                                        @elseif($distribution->is_reversed)
                                            <div>
                                                <small class="text-muted">Reversed by:</small><br>
                                                <small class="text-muted">{{ $distribution->reversedBy->name ?? 'Unknown Admin' }}</small>
                                                @if($distribution->reversal_reason)
                                                    <br><small class="text-muted">Reason: {{ $distribution->reversal_reason }}</small>
                                                @endif
                                            </div>
                                        @else
                                            <small class="text-muted">{{ $distribution->time_to_reverse }}</small>
                                        @endif
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="9" class="text-center">No ROI distributions found</td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <div class="d-flex justify-content-center">
                        {{ $distributions->appends(request()->query())->links() }}
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Reverse ROI Modal -->
<div class="modal fade" id="reverseRoiModal" tabindex="-1" aria-labelledby="reverseRoiModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="reverseRoiModalLabel">Reverse ROI Distribution</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <strong>⚠️ Warning:</strong> This action will:
                    <ul class="mb-0 mt-2">
                        <li>Deduct the ROI amount from the user's balance</li>
                        <li>Revert the investment progress</li>
                        <li>Mark the transaction as failed</li>
                        <li>This action cannot be undone!</li>
                    </ul>
                </div>
                
                <div class="mb-3">
                    <strong>User:</strong> <span id="reverseUserName"></span><br>
                    <strong>ROI Amount:</strong> <span id="reverseRoiAmount"></span><br>
                    <strong>Distribution ID:</strong> <span id="reverseDistributionId"></span>
                </div>
                
                <div class="form-group">
                    <label for="reversalReason">Reason for Reversal <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="reversalReason" rows="3" placeholder="Enter the reason for reversing this ROI distribution..." required></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="confirmReverseRoi()">
                    <i class="fas fa-undo"></i> Confirm Reversal
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function applyFilters() {
    const startDate = document.getElementById('startDate').value;
    const endDate = document.getElementById('endDate').value;
    const status = document.getElementById('statusFilter').value;
    
    let url = '{{ route("admin.roi-tracking.distribution-history") }}?';
    const params = new URLSearchParams();
    
    if (startDate) params.append('start_date', startDate);
    if (endDate) params.append('end_date', endDate);
    if (status) params.append('status', status);
    
    window.location.href = url + params.toString();
}

function resetFilters() {
    window.location.href = '{{ route("admin.roi-tracking.distribution-history") }}';
}

// Function to show reverse ROI modal
function showReverseModal(distributionId, userName, roiAmount) {
    document.getElementById('reverseDistributionId').textContent = distributionId;
    document.getElementById('reverseUserName').textContent = userName;
    document.getElementById('reverseRoiAmount').textContent = '$' + parseFloat(roiAmount).toFixed(2);
    document.getElementById('reversalReason').value = '';
    
    $('#reverseRoiModal').modal('show');
}

// Function to confirm ROI reversal
function confirmReverseRoi() {
    const distributionId = document.getElementById('reverseDistributionId').textContent;
    const reversalReason = document.getElementById('reversalReason').value.trim();
    
    if (!reversalReason) {
        Swal.fire('Error', 'Please provide a reason for the reversal', 'error');
        return;
    }
    
    Swal.fire({
        title: '⚠️ Confirm Reversal',
        text: 'Are you absolutely sure you want to reverse this ROI distribution? This action cannot be undone!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, Reverse It!',
        cancelButtonText: 'Cancel',
        confirmButtonColor: '#dc3545',
        cancelButtonColor: '#6c757d'
    }).then((result) => {
        if (result.isConfirmed) {
            Swal.fire({
                title: 'Reversing ROI...',
                text: 'Please wait while we reverse the distribution',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            fetch('/admin/roi-tracking/reverse-roi', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    distribution_id: distributionId,
                    reversal_reason: reversalReason
                })
            })
            .then(response => response.json())
            .then(data => {
                Swal.close();
                
                if (data.success) {
                    Swal.fire({
                        title: '✅ Reversed Successfully!',
                        text: data.message,
                        icon: 'success',
                        confirmButtonText: 'OK'
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire('❌ Error', data.message || 'Failed to reverse ROI', 'error');
                }
            })
            .catch(error => {
                Swal.close();
                Swal.fire('❌ Error', 'Failed to reverse ROI: ' + error.message, 'error');
            });
        }
    });
}
</script>
@endpush
