@extends('frontend::layouts.user')
@section('title')
    {{ __('Withdraw History') }}
@endsection

@section('content')
<div class="withdraw-history-container">
    <!-- Hero Section -->
    <div class="withdraw-hero">
        <div class="hero-content">
            <div class="hero-icon">📊</div>
            <h1 class="hero-title">{{ __('Withdraw History') }}</h1>
            <p class="hero-subtitle">{{ __('Track all your withdrawal requests and their current status') }}</p>
        </div>
    </div>

    <!-- Main Content -->
    <div class="withdraw-main">
        <div class="glass-card">
            <div class="card-header">
                <h3 class="card-title">📋 {{ __('Your Withdrawal History') }}</h3>
                <p class="card-subtitle">{{ __('View and track all your withdrawal transactions') }}</p>
            </div>

            @if($transactions->count() > 0)
                <div class="history-table">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>{{ __('Transaction ID') }}</th>
                                    <th>{{ __('Method') }}</th>
                                    <th>{{ __('Amount') }}</th>
                                    <th>{{ __('Fee') }}</th>
                                    <th>{{ __('Total') }}</th>
                                    <th>{{ __('Status') }}</th>
                                    <th>{{ __('Date') }}</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($transactions as $transaction)
                                <tr class="history-row">
                                    <td>
                                        <div class="txn-id">
                                            <span class="txn-code">{{ $transaction->tnx }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="method-info">
                                            <span class="method-name">{{ str_replace('SIMPLE-', '', $transaction->method) }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="amount-info">
                                            <span class="amount">{{ $transaction->pay_currency ?? setting('site_currency', 'global') }}{{ number_format($transaction->amount, 2) }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="fee-info">
                                            <span class="fee">{{ $transaction->pay_currency ?? setting('site_currency', 'global') }}{{ number_format($transaction->charge, 2) }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="total-info">
                                            <span class="total">{{ $transaction->pay_currency ?? setting('site_currency', 'global') }}{{ number_format($transaction->final_amount, 2) }}</span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="status-info">
                                            @if($transaction->status->value === 'success')
                                                <span class="status-badge success">{{ __('Completed') }}</span>
                                            @elseif($transaction->status->value === 'pending')
                                                <span class="status-badge pending">{{ __('Pending') }}</span>
                                            @elseif($transaction->status->value === 'failed')
                                                <span class="status-badge failed">{{ __('Failed') }}</span>
                                            @else
                                                <span class="status-badge {{ $transaction->status->value }}">{{ ucfirst($transaction->status->value) }}</span>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="date-info">
                                            <span class="date">{{ $transaction->created_at }}</span>
                                        </div>
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Pagination -->
                @if($transactions->hasPages())
                    <div class="pagination-wrapper">
                        {{ $transactions->links() }}
                    </div>
                @endif
            @else
                <div class="empty-state">
                    <div class="empty-icon">📭</div>
                    <h4 class="empty-title">{{ __('No Withdrawals Yet') }}</h4>
                    <p class="empty-desc">{{ __('You haven\'t made any withdrawal requests yet.') }}</p>
                    <a href="{{ route('user.withdraw.simple.index') }}" class="empty-action">
                        {{ __('Make Your First Withdrawal') }}
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
/* Withdraw History Styles */
.withdraw-history-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 20px;
}

.withdraw-hero {
    text-align: center;
    margin-bottom: 40px;
    padding: 40px 20px;
}

.hero-content {
    max-width: 600px;
    margin: 0 auto;
}

.hero-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    animation: float 3s ease-in-out infinite;
}

.hero-title {
    font-size: 3rem;
    font-weight: 700;
    color: white;
    margin-bottom: 15px;
    text-shadow: 0 4px 8px rgba(0,0,0,0.3);
}

.hero-subtitle {
    font-size: 1.2rem;
    color: rgba(255,255,255,0.9);
    line-height: 1.6;
}

.withdraw-main {
    max-width: 1200px;
    margin: 0 auto;
}

.glass-card {
    background: rgba(255,255,255,0.1);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(255,255,255,0.2);
    padding: 30px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
}

.card-header {
    text-align: center;
    margin-bottom: 30px;
}

.card-title {
    font-size: 1.8rem;
    font-weight: 600;
    color: white;
    margin-bottom: 10px;
}

.card-subtitle {
    color: rgba(255,255,255,0.8);
    font-size: 1rem;
}

/* Table Styles */
.history-table {
    margin-bottom: 30px;
}

.table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    color: white;
}

.table thead th {
    background: rgba(255,255,255,0.1);
    color: white;
    font-weight: 600;
    padding: 15px;
    text-align: left;
    border-bottom: 1px solid rgba(255,255,255,0.2);
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.table tbody tr {
    transition: all 0.3s ease;
    border-bottom: 1px solid rgba(255,255,255,0.1);
}

.table tbody tr:hover {
    background: rgba(255,255,255,0.05);
    transform: translateY(-2px);
}

.table tbody td {
    padding: 15px;
    vertical-align: middle;
}

/* Transaction ID */
.txn-id {
    display: flex;
    align-items: center;
}

.txn-code {
    background: rgba(255,255,255,0.1);
    padding: 5px 10px;
    border-radius: 8px;
    font-family: 'Courier New', monospace;
    font-size: 0.85rem;
    font-weight: 500;
    color: #4caf50;
}

/* Method Info */
.method-info {
    display: flex;
    align-items: center;
}

.method-name {
    background: rgba(102, 126, 234, 0.2);
    padding: 5px 12px;
    border-radius: 15px;
    font-size: 0.85rem;
    font-weight: 500;
    color: #667eea;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Amount Info */
.amount-info, .fee-info, .total-info {
    display: flex;
    align-items: center;
}

.amount, .fee, .total {
    font-weight: 600;
    font-size: 0.95rem;
}

.amount {
    color: #4caf50;
}

.fee {
    color: #ff9800;
}

.total {
    color: #2196f3;
}

/* Status Badge */
.status-info {
    display: flex;
    align-items: center;
}

.status-badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-badge.success {
    background: rgba(76, 175, 80, 0.2);
    color: #4caf50;
    border: 1px solid rgba(76, 175, 80, 0.3);
}

.status-badge.pending {
    background: rgba(255, 152, 0, 0.2);
    color: #ff9800;
    border: 1px solid rgba(255, 152, 0, 0.3);
}

.status-badge.failed {
    background: rgba(244, 67, 54, 0.2);
    color: #f44336;
    border: 1px solid rgba(244, 67, 54, 0.3);
}

/* Date Info */
.date-info {
    display: flex;
    align-items: center;
}

.date {
    color: rgba(255,255,255,0.8);
    font-size: 0.85rem;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 60px 20px;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    opacity: 0.7;
}

.empty-title {
    color: white;
    font-size: 1.5rem;
    font-weight: 600;
    margin-bottom: 10px;
}

.empty-desc {
    color: rgba(255,255,255,0.8);
    font-size: 1rem;
    margin-bottom: 30px;
    line-height: 1.6;
}

.empty-action {
    display: inline-block;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    text-decoration: none;
    padding: 15px 30px;
    border-radius: 15px;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.empty-action:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 35px rgba(102, 126, 234, 0.4);
    color: white;
}

/* Pagination */
.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 30px;
}

.pagination-wrapper .pagination {
    display: flex;
    gap: 10px;
}

.pagination-wrapper .page-link {
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    color: white;
    padding: 10px 15px;
    border-radius: 10px;
    text-decoration: none;
    transition: all 0.3s ease;
}

.pagination-wrapper .page-link:hover {
    background: rgba(255,255,255,0.2);
    transform: translateY(-2px);
}

.pagination-wrapper .page-item.active .page-link {
    background: #667eea;
    border-color: #667eea;
}

.pagination-wrapper .page-item.disabled .page-link {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Animations */
@keyframes float {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-10px); }
}

/* Responsive Design */
@media (max-width: 768px) {
    .withdraw-history-container {
        padding: 15px;
    }
    
    .hero-title {
        font-size: 2rem;
    }
    
    .hero-subtitle {
        font-size: 1rem;
    }
    
    .glass-card {
        padding: 20px;
    }
    
    .table {
        font-size: 0.8rem;
    }
    
    .table thead th,
    .table tbody td {
        padding: 10px 8px;
    }
    
    .txn-code {
        font-size: 0.75rem;
        padding: 3px 6px;
    }
    
    .method-name {
        font-size: 0.75rem;
        padding: 3px 8px;
    }
    
    .status-badge {
        font-size: 0.7rem;
        padding: 4px 8px;
    }
}

@media (max-width: 480px) {
    .hero-icon {
        font-size: 3rem;
    }
    
    .hero-title {
        font-size: 1.8rem;
    }
    
    .card-title {
        font-size: 1.5rem;
    }
    
    .table {
        font-size: 0.7rem;
    }
    
    .table thead th,
    .table tbody td {
        padding: 8px 5px;
    }
    
    .empty-icon {
        font-size: 3rem;
    }
    
    .empty-title {
        font-size: 1.3rem;
    }
    
    .empty-action {
        padding: 12px 25px;
        font-size: 0.9rem;
    }
}
</style>

@endsection
